/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.io.File;
import java.util.Calendar;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.chemomentum.common.ws.IServiceOrchestrator;
import org.chemomentum.workassignment.xmlbeans.CancelResultType;
import org.chemomentum.workassignment.xmlbeans.CancelResultType.Enum;
import org.chemomentum.workassignment.xmlbeans.CancelWorkAssignmentRequestDocument;
import org.chemomentum.workassignment.xmlbeans.CancelWorkAssignmentResponseDocument;
import org.chemomentum.workassignment.xmlbeans.GetWorkAssignmentStatusRequestDocument;
import org.chemomentum.workassignment.xmlbeans.GetWorkAssignmentStatusResponseDocument;
import org.chemomentum.workassignment.xmlbeans.StatusType;
import org.unigrids.x2006.x04.services.jms.TargetSystemReferenceDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.client.JobClient;
import de.fzj.unicore.uas.client.TSSClient;
import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseJob;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.exceptions.HiLANotImplementedException;
import eu.unicore.hila.grid.Grid;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.StatusChangeListener;
import eu.unicore.hila.grid.TaskStatus;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * @author bjoernh
 * 
 *         13.01.2011 13:44:46
 * 
 */
@ResourceType(locationStructure = {
	"unicore6:/orchestrators/{orchestrator}/{task}/?",
	"unicore6:/{user}@orchestrators/{orchestrator}/{task}/?" })
public class Unicore6OrchestratorJob extends BaseJob implements Job {

    private IServiceOrchestrator orchestrator;

    private Job realJob;

    private IClientConfiguration secprops;

    private static final Map<Location, Unicore6OrchestratorJob> jobsCache = new HashMap<Location, Unicore6OrchestratorJob>();

    /**
     * @param _location
     * @param _orchestrator
     */
    public Unicore6OrchestratorJob(Location _location,
	    IServiceOrchestrator _orchestrator, Unicore6SecurityProperties secprop) {
	super(_location);

	this.orchestrator = _orchestrator;
	this.secprops = secprop;

	synchronized (jobsCache) {
	    jobsCache.put(_location, this);
	}
    }

    public static synchronized Resource locate(Location _location,
	    Object... _extraInformation) throws HiLAFactoryException {
	if (jobsCache.containsKey(_location)) {
	    return jobsCache.get(_location);
	}
	throw new HiLAFactoryException("This task does not exist.");
    }

    /**
     * @see eu.unicore.hila.grid.Task#abort()
     */
    @Override
    public void abort() throws HiLAException {
	CancelWorkAssignmentRequestDocument cwareqd = CancelWorkAssignmentRequestDocument.Factory
		.newInstance();
	cwareqd.addNewCancelWorkAssignmentRequest().setId(getId());

	CancelWorkAssignmentResponseDocument cwaresd = orchestrator
		.cancelWorkAssignment(cwareqd);
	Enum result = cwaresd.getCancelWorkAssignmentResponse()
		.getCancelResult();
	if (result.equals(CancelResultType.FAILURE)) {
	    throw new HiLAException("Unable to cancel work assignment.");
	}
    }

    /**
     * @see eu.unicore.hila.grid.Task#getId()
     */
    @Override
    public String getId() throws HiLAException {
	return location.getName();
    }

    /**
     * @see eu.unicore.hila.grid.Job#startSync(java.io.File[])
     */
    @Override
    public TaskStatus startSync(File... imports) throws HiLAException {
	if (imports.length > 0) {
	    throw new HiLAException(
		    "Cannot import local files into brokered job.");
	}
	// silently ignore start, as it is done by the orchestrator
	return status();
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseTask#status()
     */
    @Override
    public TaskStatus status() throws HiLAException {
	GetWorkAssignmentStatusRequestDocument gwasreqd = GetWorkAssignmentStatusRequestDocument.Factory
		.newInstance();
	gwasreqd.addNewGetWorkAssignmentStatusRequest().setId(getId());

	GetWorkAssignmentStatusResponseDocument gwasresd = orchestrator
		.getWorkAssignmentStatus(gwasreqd);
	StatusType.Enum result = gwasresd.getGetWorkAssignmentStatusResponse()
		.getStatusResult();

	if (result.equals(StatusType.CANCELED)) {
	    return TaskStatus.ABORTED;
	} else if (result.equals(StatusType.FAILURE)) {
	    return TaskStatus.FAILED;
	} else if (result.equals(StatusType.SUCCESS)) {
	    return TaskStatus.SUCCESSFUL;
	} else if (result.equals(StatusType.RUNNING)) {
	    return TaskStatus.RUNNING;
	} else {
	    return TaskStatus.NEW;
	}
    }

    /**
     * @see eu.unicore.hila.grid.Job#startASync(java.io.File[])
     */
    @Override
    public void startASync(File... imports) throws HiLAException {
	startSync(imports);
    }

    /**
     * @see eu.unicore.hila.grid.Job#startASync(eu.unicore.hila.grid.StatusChangeListener,
     *      java.io.File[])
     */
    @Override
    public TaskStatus startASync(StatusChangeListener listener, File... imports)
	    throws HiLAException {
	startASync(imports);
	registerStatusChangeListener(listener);
	return status();
    }

    /**
     * @see eu.unicore.hila.grid.Job#cleanup(eu.unicore.hila.grid.File[])
     */
    @Override
    public List<SimpleTransfer> cleanup(eu.unicore.hila.grid.File... exports)
	    throws HiLAException {
	if (exports.length > 0) {
	    throw new HiLAException("Cannot export files from brokered job.");
	}
	throw new HiLANotImplementedException("Cleanup");
    }

    /**
     * @see eu.unicore.hila.grid.Job#getExitCode()
     */
    @Override
    public int getExitCode() throws HiLAException {
	if (getRealJob() == null) {
	    throw new HiLAException("exit code not accessible");
	}
	return realJob.getExitCode();
    }

    /**
     * @see eu.unicore.hila.grid.Job#getLog()
     */
    @Override
    public String getLog() throws HiLAException {
	if (getRealJob() == null) {
	    throw new HiLAException("Job log not accessible");
	}
	return realJob.getLog();
    }
    
    /**
     * @see eu.unicore.hila.grid.Job#getStdErr()
     */
    @Override
    public eu.unicore.hila.grid.File getStdErr() throws HiLAException {
	if (getRealJob() == null) {
	    throw new HiLAException("stderr not accessible");
	}
	return null;
    }

    /**
     * @see eu.unicore.hila.grid.Job#getStdOut()
     */
    @Override
    public eu.unicore.hila.grid.File getStdOut() throws HiLAException {
	if (getRealJob() == null) {
	    throw new HiLAException("stdout not accessible");
	}
	return realJob.getStdOut();
    }

    /**
     * @see eu.unicore.hila.grid.Job#getSubmissionTime()
     */
    @Override
    public Calendar getSubmissionTime() throws HiLAException {
	if(getRealJob() == null) {
	    throw new HiLAException("Job submission time not accessible.");
	}
	return realJob.getSubmissionTime();
    }

    /**
     * @see eu.unicore.hila.grid.Job#getWorkingDirectory()
     */
    @Override
    public eu.unicore.hila.grid.File getWorkingDirectory() throws HiLAException {
	if (getRealJob() == null) {
	    throw new HiLAException("Working directory not accessible.");
	}
	return realJob.getWorkingDirectory();
    }

    private synchronized Job getRealJob() throws HiLAException {
	if (realJob == null) {
	    GetWorkAssignmentStatusRequestDocument gwasreqd = GetWorkAssignmentStatusRequestDocument.Factory
		    .newInstance();
	    gwasreqd.addNewGetWorkAssignmentStatusRequest().setId(getId());

	    GetWorkAssignmentStatusResponseDocument gwasresd = orchestrator
		    .getWorkAssignmentStatus(gwasreqd);
	    if (gwasresd.getGetWorkAssignmentStatusResponse()
		    .isSetExecutionResources()) {
		EndpointReferenceType jobEpr = gwasresd
			.getGetWorkAssignmentStatusResponse()
			.getExecutionResources().getJob();

		realJob = getJobFromEpr(jobEpr);
	    }
	}
	return realJob;
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseJob#getChildren()
     */
    @Override
    public List<Resource> getChildren() throws HiLAException {
	if (getRealJob() == null) {
	    throw new HiLAException("No children available (yet).");
	}
	return Collections.singletonList((Resource) getRealJob());
    }

    /**
     * @param jobEpr
     * @return
     * @throws HiLAException
     */
    private Job getJobFromEpr(EndpointReferenceType jobEpr)
	    throws HiLAException {
	try {
	    JobClient jobClient = new JobClient(jobEpr.getAddress()
		    .getStringValue(), jobEpr, secprops);
	    TargetSystemReferenceDocument tsrd = TargetSystemReferenceDocument.Factory
		    .parse(jobClient
			    .getResourceProperty(TargetSystemReferenceDocument.type
				    .getDocumentElementName()));
	    TSSClient tssClient = new TSSClient(tsrd.getTargetSystemReference()
		    .getAddress().getStringValue(),
		    tsrd.getTargetSystemReference(), secprops);

	    Calendar upSince = tssClient.getUpSince();
	    String dateString = GetSitesFromTSF.constructDateString(upSince);
	    String tsName = tssClient.getTargetSystemName();

	    Location gridLoc = findParentLocationOfType(Grid.class, location,
		    Grid.class);
	    Job job = (Job) gridLoc.getChildLocation("sites")
		    .getChildLocation(tsName.concat("_").concat(dateString))
		    .getChildLocation("tasks")
		    .getChildLocation(Unicore6Site.epr2TaskId(jobEpr)).locate();
	    return job;
	} catch (Exception e) {
	    throw new HiLAException("Unable to access the real job.", e);
	}
    }
}
