/*********************************************************************************
 * Copyright (c) 2009-2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import org.apache.xmlbeans.XmlException;
import org.unigrids.x2006.x04.services.jms.LogDocument;
import org.unigrids.x2006.x04.services.jms.OriginalJSDLDocument;

import de.fzj.unicore.uas.JobManagement;
import de.fzj.unicore.uas.client.JobClient;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.wsrflite.xfire.ClientException;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.InvalidResourcePropertyQNameFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnavailableFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnknownFault;
import eu.unicore.hila.HiLA;
import eu.unicore.hila.Location;
import eu.unicore.hila.Metadata;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseJob;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationPatternException;
import eu.unicore.hila.exceptions.HiLAResourceAlreadyExistsException;
import eu.unicore.hila.exceptions.HiLAResourceNotFoundException;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.Site;
import eu.unicore.hila.grid.TaskStatus;

/**
 * @author bjoernh
 * 
 *         14.08.2009 15:59:11
 * 
 */
@ResourceType(locationStructure = { "unicore6:/sites/{site}/tasks/{task}/?",
	"unicore6:/{user}@sites/{site}/tasks/{task}/?" })
public class Unicore6Job extends BaseJob implements Job {

    private static final Cache tasksCache;
    static {
	final CacheManager manager = CacheManager.getInstance();
	manager.addCache(new Cache(Unicore6Job.class.getName(), 10000, true,
		true, 0, 0));
	tasksCache = manager.getCache(Unicore6Job.class.getName());
    }

    protected final JobClient jobClient;

    private Metadata metadata;

    private static Map<String, TaskStatus> statusMap;
    static {
	statusMap = new Hashtable<String, TaskStatus>();

	statusMap.put("READY", TaskStatus.NEW);
	statusMap.put("UNDEFINED", TaskStatus.NEW);
	statusMap.put("STAGINGIN", TaskStatus.NEW);
	statusMap.put("QUEUED", TaskStatus.PENDING);
	statusMap.put("RUNNING", TaskStatus.RUNNING);
	statusMap.put("STAGINGOUT", TaskStatus.RUNNING);
	statusMap.put("SUCCESSFUL", TaskStatus.SUCCESSFUL);
	statusMap.put("FAILED", TaskStatus.FAILED);
    }

    Unicore6Job(Location _location, JobClient _jobClient)
	    throws HiLAException {
	super(_location);

	if (tasksCache.isKeyInCache(_location)) {
	    throw new HiLAResourceAlreadyExistsException();
	}

	this.jobClient = _jobClient;
	synchronized (tasksCache) {
	    tasksCache.put(new Element(_location, this));
	}
    }

    public static Resource locate(Location _location,
	    Object... _extraInformation) throws HiLAException {
	if (tasksCache.isKeyInCache(_location)) {
	    Element cacheElem = tasksCache.get(_location);
	    if (cacheElem != null) {
		return (Resource) cacheElem.getObjectValue();
	    }
	}
	return ((Unicore6Site) findParentLocationOfType(Unicore6Site.class,
		_location, Unicore6Grid.class).locate()).getTask(_location
		.getName());

    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Task#abort()
     */
    public void abort() throws HiLAException {
	try {
	    jobClient.abort();
	} catch (Exception e) {
	    throw new HiLAException("Job abort failed", e);
	}
    }

    /**
     * @see eu.unicore.hila.grid.Task#getId()
     */
    public String getId() {
	return this.location.getName();
    }

    /**
     * @throws HiLAException
     * @throws HiLALocationPatternException
     * @see eu.unicore.hila.grid.Task#getSite()
     */
    public Site getSite() throws HiLALocationPatternException, HiLAException {
	return (Site) findParentLocationOfType(Unicore6Site.class,
		this.location, Unicore6Grid.class).locate();
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Task#status()
     */
    public TaskStatus status() throws HiLAException {
        if ((super.status() != null) && super.status().isFinalState()) {
	    return super.status();
	} else {
	    return updateStatus();
	}
    }

    /**
     * @return
     * @throws HiLAException
     */
    private TaskStatus updateStatus() throws HiLAException {
	try {
	    String jobStatus = jobClient.getStatus().toString();
	    if (jobStatus != null) {
		TaskStatus status = statusMap.get(jobStatus);
		setCurrentState(status);
		return status;
	    } else {
		throw new HiLAException("Unable to determine Task status: null");
	    }
	} catch (Exception e) {
	    throw new HiLAException("Unable to determine Task status", e);
	}
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Task#startASync(java.io.File[])
     */
    public void startASync(File... files) throws HiLAException {
	try {
	    List<SimpleTransfer> imports = new ArrayList<SimpleTransfer>();
	    for (File file : files) {
		imports.add(this.getWorkingDirectory()
			.importFromLocalFile(file));
	    }

	    boolean allDone;
	    do {
		allDone = true;
		for (SimpleTransfer simpleTransfer : imports) {
		    allDone = allDone
			    && simpleTransfer.status().equals(
				    TaskStatus.SUCCESSFUL);
		}
	    } while (!allDone);

	    jobClient.start();
	} catch (Exception e) {
	    HiLAException e2 = new HiLAException("startASync of Task failed.",
		    e);
	    setCurrentState(TaskStatus.FAILED, e2);
	    throw e2;
	}

    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Job#cleanup(eu.unicore.hila.grid.File[])
     */
    public List<SimpleTransfer> cleanup(eu.unicore.hila.grid.File... exports)
	    throws HiLAException {

	List<SimpleTransfer> transfers = new ArrayList<SimpleTransfer>();

	if (exports != null && exports.length > 0) {
	    // TODO define property
	    File localDataDir = new File(new File(
		    System.getProperty("user.home")), ".hila2/data");
	    if (!localDataDir.exists()) {
		if (!localDataDir.mkdirs()) {
		    throw new HiLAException(
			    "Unable to create local directory for exports: "
				    + localDataDir.getAbsolutePath());
		}
	    } else if (!localDataDir.isDirectory()) {
		throw new HiLAException("Local file "
			+ localDataDir.getAbsolutePath()
			+ " blocks task output directory.");
	    }
	    File localDir = new File(localDataDir, this.getId());
	    if (!localDir.mkdir()) {
		throw new HiLAException(
			"Unable to create local directory for export: "
				+ localDir.getAbsolutePath());
	    }
	    for (eu.unicore.hila.grid.File exportFile : exports) {
		transfers.add(exportFile.exportToLocalFile(localDir));
	    }
	}

      CleanupAction cleanup = new CleanupAction(transfers, this);
      HiLA.getExecutor().submit(cleanup);

	return transfers;
    }

    /**
     * @see eu.unicore.hila.grid.Job#getExitCode()
     */
    public int getExitCode() {
	return jobClient.getExitCode();
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Job#getLog()
     */
    public String getLog() throws HiLAException {
	try {
	    LogDocument ld = LogDocument.Factory.parse(jobClient
		    .getResourceProperty(JobManagement.RPLog));
	    return ld.getLog();
	} catch (InvalidResourcePropertyQNameFault e) {
	    throw new HiLAException("Unable to get job log.", e);
	} catch (BaseFault e) {
	    throw new HiLAException("Unable to get job log.", e);
	} catch (ResourceUnavailableFault e) {
	    throw new HiLAException("Unable to get job log.", e);
	} catch (ResourceUnknownFault e) {
	    throw new HiLAException("Unable to get job log.", e);
	} catch (ClientException e) {
	    throw new HiLAException("Unable to get job log.", e);
	} catch (XmlException e) {
	    throw new HiLAException("Unable to parse job log.", e);
	}
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Job#getStdErr()
     */
    public eu.unicore.hila.grid.File getStdErr() throws HiLAException {
	return (eu.unicore.hila.grid.File) this.getWorkingDirectory().getChild(
		"stderr");
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Job#getStdOut()
     */
    public eu.unicore.hila.grid.File getStdOut() throws HiLAException {
	return (eu.unicore.hila.grid.File) this.getWorkingDirectory().getChild(
		"stdout");
    }

    /**
     * @see eu.unicore.hila.grid.Job#getSubmissionTime()
     */
    @Override
    public Calendar getSubmissionTime() throws HiLAException {
	return jobClient.getSubmissionTime();
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.common.grid.BaseJob#getTaskName()
     */
    @Override
    public String getTaskName() throws HiLAException {
	try {
	    OriginalJSDLDocument jsdl = OriginalJSDLDocument.Factory
		    .parse(jobClient
			    .getResourceProperty(OriginalJSDLDocument.type
				    .getDocumentElementName()));
	    if (jsdl.getOriginalJSDL().getJobDescription()
		    .isSetJobIdentification()) {
		if (jsdl.getOriginalJSDL().getJobDescription()
			.getJobIdentification().isSetJobName()) {
		    return jsdl.getOriginalJSDL().getJobDescription()
			    .getJobIdentification().getJobName();
		}
	    }
	    return super.getTaskName();
	} catch (InvalidResourcePropertyQNameFault e) {
	    throw new HiLAException("Unknown Resource property.", e);
	} catch (BaseFault e) {
	    throw new HiLAException("Unable to retrieve task name.", e);
	} catch (ResourceUnavailableFault e) {
	    throw new HiLAResourceNotFoundException(
		    "Job resource unavailable.", e);
	} catch (ResourceUnknownFault e) {
	    throw new HiLAResourceNotFoundException(
		    "Job resource unavailable.", e);
	} catch (ClientException e) {
	    throw new HiLAException("Unable to retrieve task name.", e);
	} catch (XmlException e) {
	    throw new HiLAException(
		    "Server returned invalid resource property.", e);
	}
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Job#getWorkingDirectory()
     */
    public eu.unicore.hila.grid.File getWorkingDirectory() throws HiLAException {
	try {
	    StorageClient wdStorageClient = jobClient.getUspaceClient();
	    final Location wdLoc = location.getChildLocation("wd")
		    .getChildLocation("files");
	    return (eu.unicore.hila.grid.File) wdLoc.locate(wdStorageClient);
	} catch (Exception e) {
	    throw new HiLAException(
		    "Unable to access working directory of task " + location
			    + ".", e);
	}
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseJob#hold()
     */
    @Override
    public void hold() throws HiLAException {
	try {
	    jobClient.hold();
	} catch (Exception e) {
	    throw new HiLAException("cannot hold job", e);
	}
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseJob#resume()
     */
    @Override
    public void resume() throws HiLAException {
	try {
	    jobClient.resume();
	} catch (Exception e) {
	    throw new HiLAException("cannot resume job", e);
	}
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseJob#getMetadata()
     */
    @Override
    public Metadata getMetadata() throws HiLAException {
        if(this.metadata == null) {
            this.metadata = new Metadata();
        }
        try {
	    this.metadata.setData("resourceProperties", jobClient.getResourcePropertyDocument());
	} catch (BaseFault e) {
	    throw new HiLAException("resourceProperties unavailable", e);
	} catch (ResourceUnknownFault e) {
	    throw new HiLAResourceNotFoundException("resourceProperties unavailable", e);
	} catch (ResourceUnavailableFault e) {
	    throw new HiLAResourceNotFoundException("resourceProperties unavailable", e);
	} catch (ClientException e) {
	    throw new HiLAException("resourceProperties unavailable", e);
	}
	return this.metadata;
    }

   /**
    * @param job
    */
   protected static void removeFromCache(Unicore6Job job) {
      tasksCache.remove(job.location);

   }
}
