/*********************************************************************************
 * Copyright (c) 2009-2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Callable;

import de.fzj.unicore.uas.client.FileTransferClient;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.wsrflite.xfire.ClientException;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceNotDestroyedFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnavailableFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnknownFault;
import eu.unicore.hila.HiLA;
import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseSimpleTransfer;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLANotImplementedException;
import eu.unicore.hila.exceptions.HiLAResourceNotFoundException;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.TaskStatus;
import eu.unicore.hila.grid.unicore6.util.FileTransfers;

/**
 * @author bjoernh
 * 
 *         27.10.2009 12:40:44
 * 
 */
@ResourceType(locationStructure = {
	"unicore6:/sites/{site}/storages/{storage}/exports/{export}/?",
	"unicore6:/{user}@sites/{site}/storages/{storage}/exports/{export}/?",
	"unicore6:/sites/{site}/tasks/{task}/wd/exports/{export}",
	"unicore6:/{user}@sites/{site}/tasks/{task}/wd/exports/{export}",
	"unicore6:/storages/{storage}/exports/{export}",
	"unicore6:/{user}@storages/{storage}/exports/{export}" })
public class Unicore6ExportTask extends BaseSimpleTransfer implements
	SimpleTransfer, Callable<TaskStatus> {
    private StorageClient storageClient;
    private String path;
    private eu.unicore.hila.grid.File remoteFile;
    private File localFile;
    private boolean overwrite;
    private FileTransferClient exportClient;
    private long maxBytes;
    private long currentBytes;

    /**
     * @param _storageLocation
     * @throws HiLAException
     */
    public Unicore6ExportTask(Location _storageLocation,
	    StorageClient _storageClient, String _path,
	    eu.unicore.hila.grid.File _remoteFile, File _localFile,
	    boolean _overWrite) throws HiLAException {
	super(_storageLocation.getChildLocation("exports").getChildLocation(
		UUID.randomUUID().toString()));

	this.storageClient = _storageClient;
	this.path = _path;
	this.remoteFile = _remoteFile;
	this.localFile = _localFile;
	this.overwrite = _overWrite;

	setCurrentState(TaskStatus.PENDING);

      HiLA.getExecutor().submit(this);

	Unicore6ExportsCollection ec = (Unicore6ExportsCollection) this
		.getParent();
	if (ec != null) {
	    ec.addTask(this);
	}
    }

    /**
     * @param childLocation
     */
    protected Unicore6ExportTask(Location _location) {
	super(_location);
    }

    public static synchronized Unicore6ExportTask locate(Location _location,
	    Object... _extraInformation) throws HiLAException {
	Resource exportsRes = _location.getParentLocation().locate();
	List<Resource> exports = exportsRes.getChildren();
	for (Resource resource : exports) {
	    if (resource.getLocation().equals(_location)) {
		return (Unicore6ExportTask) resource;
	    }
	}
	throw new HiLAResourceNotFoundException("This export does not exist: "
		+ _location);
    }

    /**
     * @see java.util.concurrent.Callable#call()
     */
    @Override
    public TaskStatus call() throws Exception {
	setCurrentState(TaskStatus.RUNNING);
	if (this.storageClient != null) {
	    OutputStream fos = null;
	    try {
		if (!this.remoteFile.exists()) {
		    HiLAException e = new HiLAException(
			    "No such file or directory.");
		    setCurrentState(TaskStatus.FAILED, e);
		    throw e;
		}

            // needed for UFTP
            final Map<String, String> extraParameters = new HashMap<String, String>();
            FileTransfers
                  .getDefaultUftpParameters(extraParameters);

            this.exportClient = this.storageClient.getExport(this.path,
                  extraParameters, FileTransfers.getFTPrefs());
		if (this.overwrite || !this.localFile.exists()) {
		    fos = new FileOutputStream(this.localFile);
		    exportClient.readAllData(fos);
		    fos.close();
		} else {
		    HiLAException e = new HiLAException(
			    "File exists, but may not be overwritten.");
		    setCurrentState(TaskStatus.FAILED, e);
		    exportClient.destroy();
		    exportClient = null;
		    throw e;
		}
	    } catch (IOException e) {
		HiLAException e2 = new HiLAException(
			"Unable to export to local file.", e);
		setCurrentState(TaskStatus.FAILED, e2);
		if (exportClient != null) {
		    exportClient.destroy();
		    exportClient = null;
		}
		throw e2;
	    } catch (Exception e) {
		HiLAException e2 = new HiLAException(
			"Exception while reading data from remote file.", e);
		setCurrentState(TaskStatus.FAILED, e2);
		if (exportClient != null) {
		    exportClient.destroy();
		    exportClient = null;
		}
		throw e2;
	    } finally {
		if (fos != null) {
		    try {
			fos.close();
		    } catch (IOException e) {
			// at least we've tried
		    }
		}
	    }
	}

	setCurrentState(TaskStatus.SUCCESSFUL);
	exportClient.destroy();
	exportClient = null;
	return status();
    }

    /**
     * @see eu.unicore.hila.grid.Task#abort()
     */
    public synchronized void abort() throws HiLAException {
	try {
	    if (exportClient != null) {
		exportClient.destroy();
		exportClient = null;
	    }
	} catch (BaseFault e) {
	    throw new HiLAException(e.getMessage(), e);
	} catch (ResourceUnavailableFault e) {
	    throw new HiLAException(e.getMessage(), e);
	} catch (ResourceUnknownFault e) {
	    throw new HiLAException(e.getMessage(), e);
	} catch (ResourceNotDestroyedFault e) {
	    throw new HiLAException(e.getMessage(), e);
	} catch (ClientException e) {
	    throw new HiLAException(e.getMessage(), e);
	}
    }

    /**
     * @see eu.unicore.hila.grid.Task#getId()
     */
    public String getId() {
	return location.getName();
    }

    /**
     * @see eu.unicore.hila.Resource#getChildren()
     */
    public List<Resource> getChildren() throws HiLAException {
	final List<Resource> children = new ArrayList<Resource>();
	return children;
    }

    // Progressable implementation
    /**
     * @see eu.unicore.hila.common.grid.BaseTask#getUnit()
     */
    @Override
    public String getUnit() throws HiLANotImplementedException {
	return "bytes";
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseTask#getMax()
     */
    @Override
    public synchronized long getMax() throws HiLAException {
	if (exportClient != null) {
	    this.maxBytes = exportClient.getSourceFileSize();
	}
	return this.maxBytes;
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseTask#getCurrent()
     */
    @Override
    public synchronized long getCurrent() throws HiLAException {
	if (exportClient != null) {
	    try {
		this.currentBytes = exportClient.getTransferredBytes();
	    } catch (Exception e) {
		throw new HiLAException(
			"Unable to determine current transfer status.", e);
	    }
	}
	return this.currentBytes;
    }

}
