/*********************************************************************************
 * Copyright (c) 2010 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;

import eu.unicore.hila.HiLA;
import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.common.BaseResource;
import eu.unicore.hila.common.grid.BaseTask;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.TaskStatus;

/**
 * @author bjoernh
 * 
 *         18.11.2010 11:27:59
 * 
 */
public class LocalTasksCollection extends BaseResource {

    private static final Logger log = Logger
	    .getLogger(LocalTasksCollection.class);

    protected final List<BaseTask> tasks;
    protected final Map<BaseTask, Long> finishedTasks;
    protected final LocalTasksCleaner tasksCleaner;
    final private Object lock = new Object();

    private boolean cleaningActive = false;
    private ScheduledFuture<?> tasksCleanerFuture;

    /**
     * @param _location
     */
    public LocalTasksCollection(Location _location) {
	super(_location);

	tasks = new ArrayList<BaseTask>();
	finishedTasks = new HashMap<BaseTask, Long>();
	tasksCleaner = new LocalTasksCleaner();
    }

    /**
     * @see eu.unicore.hila.Resource#getChildren()
     */
    @Override
    public List<Resource> getChildren() throws HiLAException {
	List<Resource> children = new ArrayList<Resource>();
	synchronized (lock) {
	    children.addAll(tasks);
	    children.addAll(finishedTasks.keySet());
	}
	return children;
    }

    public void addTask(BaseTask _task) {
	synchronized (lock) {
	    tasks.add(_task);
	    if (!cleaningActive) {
		if (log.isDebugEnabled()) {
		    log.debug("Starting periodic cleanup of local tasks for "
			    + this.getLocation());
		}
		tasksCleanerFuture = HiLA.getDaemonExecutor().scheduleWithFixedDelay(
			tasksCleaner, 0, 20, TimeUnit.SECONDS);
		cleaningActive = true;
	    }
	}
    }

    public void removeTask(BaseTask _task) {
	synchronized (lock) {
	    if (tasks.contains(_task)) {
		tasks.remove(_task);
	    } else if (finishedTasks.containsKey(_task)) {
		finishedTasks.remove(_task);
	    }

	    if (tasks.isEmpty() && finishedTasks.isEmpty()) {
		if (log.isDebugEnabled()) {
		    log.debug("Stopping periodic cleanup of local tasks for "
			    + this.getLocation());
		}
		tasksCleanerFuture.cancel(false);
		cleaningActive = false;
	    }
	}
    }

    private class LocalTasksCleaner implements Runnable {

	/**
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
	    if (log.isDebugEnabled()) {
		log.debug("Running periodic cleanup of local tasks.");
	    }
	    // move finished tasks to finishedTasks list
	    final List<BaseTask> remove = new ArrayList<BaseTask>();
	    synchronized (lock) {
		for (BaseTask task : tasks) {
		    try {
			if (task.status().equals(TaskStatus.SUCCESSFUL)
				|| task.status().equals(TaskStatus.FAILED)
				|| task.status().equals(TaskStatus.ABORTED)) {
			    finishedTasks.put(task, System.currentTimeMillis());
			    remove.add(task);
			}
		    } catch (HiLAException e) {
			log.error("Must not happen.", e);
		    }
		}
		tasks.removeAll(remove);

		remove.clear();
		for (BaseTask task : finishedTasks.keySet()) {
		    if ((System.currentTimeMillis() - finishedTasks.get(task)
			    .doubleValue()) > 300000) {
			remove.add(task);
		    }
		}
		for (BaseTask task : remove) {
		    if (log.isDebugEnabled()) {
			log.debug("Removing " + task.getLocation()
				+ " from list of local tasks.");
		    }
		    removeTask(task);
		}
		if(log.isTraceEnabled()) {
		    log.trace("Active queue  : " + Integer.toString(tasks.size()));
		    log.trace("Finished queue: " + Integer.toString(finishedTasks.size()));
		}
	    }
	}

    }
}
