/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.shell.commands;

import java.io.File;
import java.io.IOException;
import java.util.Calendar;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionDocument;

import eu.unicore.hila.Location;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.Submittable;
import eu.unicore.hila.job.jsdl.JSDLtoModel;
import eu.unicore.hila.job.model.JobModel;
import eu.unicore.hila.shell.ShellContext;

/**
 * @author bjoernh
 * 
 *         16.10.2009 15:53:37
 * 
 */
public class Submit extends ShellCommandBase implements ShellCommand {

    static final Logger log = Logger.getLogger(Submit.class);

    /**
     * 
     */
    public Submit() {
	initializeOptions();
    }

    /**
     * 
     */
    private void initializeOptions() {
	options.addOption(OptionConstants.JSDL);
	options.addOption(OptionConstants.SITE);
	options.addOption(OptionBuilder.withArgName("Run").withLongOpt("run")
		.withDescription("Run job right away.").hasArg(false)
		.create("r"));
	options.addOption(OptionBuilder.withArgName("help").withLongOpt("help")
		.withDescription("Print help.").create("h"));
    }

    /**
     * @see eu.unicore.hila.shell.commands.ShellCommand#execute(java.lang.String[],
     *      eu.unicore.hila.shell.ShellContext)
     */
    public void execute(String[] cmdArgs, ShellContext ctx) {
	try {
	    CommandLine cmdline = parser.parse(options, cmdArgs);

	    if (cmdline.hasOption("h")) {
		printHelp(ctx);
		return;
	    }

	    JobModel jm = null;
	    if (cmdline.hasOption("j")) {
		String fileName = cmdline.getOptionValue("j");
		File jsdlFile = new File(fileName);
		if (!jsdlFile.isAbsolute()) {
		    jsdlFile = new File(ctx.getCurrentLocalDir(), fileName);
		}
		if (!jsdlFile.exists() || jsdlFile.isDirectory()) {
		    try {
			ctx.getCs().printString(
				"JSDL file does not exist or is a directory.");
			ctx.getCs().printNewline();
		    } catch (IOException e) {
			log.info("JSDL file does not exist or is a directory.",
				e);
		    }
		}
		try {
		    JobDefinitionDocument jdd = JobDefinitionDocument.Factory.parse(jsdlFile);
		    jm = new JSDLtoModel(jdd).getModel();
		} catch (XmlException e) {
		    log.error("Error parsing JSDL document.", e);
		} catch (IOException e) {
		    log.error("Error opening JSDL document.", e);
		}

	    } else {
		missingArg("j");
		return;
	    }

			if (jm.getTaskName() == null) {
				jm.setTaskName("HiLA Shell 'submit' job submitted at "
						+ Calendar.getInstance().getTime());
			}

	    Submittable site = determineSite(ctx, cmdline);
	    if (site == null) {
		log.error("No site given. I do not know where to submit to.");
		return;
	    }

	    try {
		Job task = site.submit(jm);
		try {
		    ctx.getCs().printString(task.getLocation().toString());
		    ctx.getCs().printNewline();
		} catch (IOException e) {
		    log.error("Unable to print using jLine.", e);
		}

		if (task != null) {
		    if (cmdline.hasOption("r")) {
			if (cmdline.hasOption("sync")) {
			    task.startSync();
			} else {
			    task.startASync();
			}
		    }
		}
	    } catch (HiLAException e) {
		log.error("Error during submission or start of job.", e);
	    }

	} catch (ParseException e) {
	    log.error("Commandline parsing failed. Not doing anything.", e);
	}
    }

    /**
     * @see eu.unicore.hila.shell.commands.ShellCommandBase#help()
     */
    @Override
    public String help() {
	StringBuilder sb = new StringBuilder();
	sb.append("Submit jobs to sites" + System.getProperty("line.separator")
		+ System.getProperty("line.separator"));
	sb.append(super.help());
	return sb.toString();
    }

    /**
     * @param ctx
     * @param cmdline
     * @return
     */
    private Submittable determineSite(ShellContext ctx, CommandLine cmdline) {
	Submittable site = null;

	if (ctx.getCurrentLocation() != null) {
	    if (ctx.getCurrentLocation().isLocationOfType(Submittable.class)) {
		try {
		    site = (Submittable) ctx.getCurrentLocation().locate();
		} catch (HiLAException e) {
		    log.error(
			    "Site not locatable, although it is the current location.",
			    e);
		}
	    }
	}

	if (cmdline.hasOption("s")) {
	    Location siteLocation = null;
	    try {
		siteLocation = new Location(cmdline.getOptionValue("s"));
		site = (Submittable) siteLocation.locate();
	    } catch (HiLAException e) {
		log.error("Given site " + siteLocation + " cannot be located.",
			e);
	    }
	}

	return site;
    }

}
