/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.job.emi;

import org.ggf.schemas.jsdl.x2005.x11.jsdl.CPUArchitectureType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ProcessorArchitectureEnumeration;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesType;

import eu.emi.es.x2010.x12.adl.ActivityDescriptionDocument;
import eu.emi.es.x2010.x12.adl.ActivityDescriptionDocument.ActivityDescription;
import eu.emi.es.x2010.x12.adl.InputFileDocument.InputFile;
import eu.emi.es.x2010.x12.adl.OptionType;
import eu.emi.es.x2010.x12.adl.OutputFileDocument.OutputFile;
import eu.emi.es.x2010.x12.adl.ResourcesDocument.Resources;
import eu.unicore.hila.job.model.JobModel;
import eu.unicore.hila.job.model.StageIn;
import eu.unicore.hila.job.model.StageOut;
import eu.unicore.hila.job.spi.NativeToModel;

/**
 * @author bjoernh
 * 
 *         21.12.2011 12:43:07
 * 
 */
public class EmiAdlToModel implements NativeToModel {

   private ActivityDescriptionDocument adl;
   private JobModel model = null;


   public EmiAdlToModel(Object _adl) {
      this.adl = (ActivityDescriptionDocument) _adl;
   }

   /**
    * @see eu.unicore.hila.job.spi.NativeToModel#getModel()
    */
   @Override
   public synchronized JobModel getModel() {
      if (model == null) {
         createModel();
      }

      return model;
   }

   /**
    * 
    */
   private void createModel() {
      this.model = new JobModel();

      ActivityDescription activity = adl.getActivityDescription();

      // ActivityIdentification is similar to JobIdentification.
      if (activity.getActivityIdentification() != null) {
         if (activity.getActivityIdentification().isSetDescription()) {
            model.setJobDescription(activity.getActivityIdentification()
                  .getDescription());
         }
         if (activity.getActivityIdentification().isSetName()) {
            model.setTaskName(activity.getActivityIdentification().getName());
         }
         if (activity.getActivityIdentification().getAnnotationArray() != null
               && activity.getActivityIdentification().getAnnotationArray().length > 0) {
            for (String annotation : activity.getActivityIdentification()
                  .getAnnotationArray()) {
               model.addJobAnnotation(annotation);
            }
         }
      }

      // application
      if (activity.isSetApplication()
            && activity.getApplication().isSetExecutable()) {
         model.setExecutable(activity.getApplication().getExecutable().getPath());
         // arguments
         if (activity.getApplication().getExecutable().getArgumentArray() != null
               && activity.getApplication().getExecutable().getArgumentArray().length > 0) {
            for (String argument : activity.getApplication().getExecutable()
                  .getArgumentArray()) {
               model.addArgument(argument);
            }
         }
         // environment
         if (activity.getApplication().getEnvironmentArray() != null
               && activity.getApplication().getEnvironmentArray().length > 0) {
            for (OptionType option : activity.getApplication()
                  .getEnvironmentArray()) {
               model.addEnvironment(option.getName(), option.getValue());
            }
         }
      }

		if (activity.isSetApplication()) {
			if (activity.getApplication().getOutput() != null) {
				model.setStdout(activity.getApplication().getOutput());
			}
			if (activity.getApplication().getError() != null) {
				model.setStderr(activity.getApplication().getError());
			}
			if (activity.getApplication().getInput() != null) {
				model.setStdin(activity.getApplication().getInput());
			}
		}

      // datastaging
      // TODO the model needs to be extended to fully support ADL datastaging
      if (activity.isSetDataStaging()) {
         if (activity.getDataStaging().getInputFileArray() != null
               && activity.getDataStaging().getInputFileArray().length > 0) {
            for (InputFile inputFile : activity.getDataStaging()
                  .getInputFileArray()) {
               model.addStageIn(new StageIn(inputFile.getName(), inputFile
                     .getSourceArray(0).getURI(), null, null, false));
            }
         }

         if (activity.getDataStaging().getOutputFileArray() != null
               && activity.getDataStaging().getOutputFileArray().length > 0) {
            for (OutputFile outputFile : activity.getDataStaging()
                  .getOutputFileArray()) {
               model.addStageOut(new StageOut(outputFile.getName(), outputFile
                     .getTargetArray(0).getURI(), null, null, false));
            }
         }

      }

      // resources
      if (activity.isSetResources()) {
         Resources res = activity.getResources();
         ResourcesType jsdlRes = ResourcesType.Factory.newInstance();

         if (res.isSetBenchmark()) {
            // TODO
         }
         if (res.isSetCoprocessor()) {
            // TODO
         }
         if (res.isSetDiskSpaceRequirement()) {

         }
         if (res.isSetIndividualCPUTime()) {
            jsdlRes.addNewIndividualCPUTime().addNewExact()
                  .setDoubleValue(res.getIndividualCPUTime().doubleValue());
         }
         if (res.isSetIndividualPhysicalMemory()) {
            jsdlRes
                  .addNewIndividualPhysicalMemory()
                  .addNewExact()
                  .setDoubleValue(
                        res.getIndividualPhysicalMemory().doubleValue());
         }
         if (res.isSetIndividualVirtualMemory()) {
            jsdlRes
                  .addNewIndividualVirtualMemory()
                  .addNewExact()
                  .setDoubleValue(
                        res.getIndividualVirtualMemory().doubleValue());
         }
         if (res.isSetNetworkInfo()) {
            // TODO
         }
         if (res.isSetNodeAccess()) {
            // TODO
         }
         if (res.isSetParallelEnvironment()) {
            // TODO
         }
         if (res.isSetPlatform()) {
            jsdlRes.setCPUArchitecture(mapPlatform(res.getPlatform()));
         }
         if (res.isSetQueueName()) {
            // TODO
         }
         if (res.isSetRemoteSessionAccess()) {
            // TODO
         }
         if (res.isSetSlotRequirement()) {
            // TODO
         }
         if (res.isSetTotalCPUTime()) {
            jsdlRes.addNewTotalCPUTime().addNewExact()
                  .setDoubleValue(res.getTotalCPUTime().doubleValue());
         }
         if (res.isSetWallTime()) {
            jsdlRes.addNewIndividualCPUTime().addNewExact()
                  .setDoubleValue(res.getWallTime().doubleValue());
         }

         model.setResources(jsdlRes);
      }

   }

   /**
    * @param platform
    * @return
    */
   private CPUArchitectureType mapPlatform(String platform) {
      CPUArchitectureType cpuArch = CPUArchitectureType.Factory.newInstance();
      if (platform.equals("amd64")) {
         cpuArch.setCPUArchitectureName(ProcessorArchitectureEnumeration.X_86_64);
      } else if (platform.equals("itanium")) {
         cpuArch.setCPUArchitectureName(ProcessorArchitectureEnumeration.IA_64);
      } else if (platform.equals("i386")) {
         cpuArch.setCPUArchitectureName(ProcessorArchitectureEnumeration.X_86);
      } else if (platform.equals("powerpc")) {
         cpuArch
               .setCPUArchitectureName(ProcessorArchitectureEnumeration.POWERPC);
      } else if (platform.equals("sparc")) {
         cpuArch.setCPUArchitectureName(ProcessorArchitectureEnumeration.SPARC);
      } else {
         // TODO this is potentially dangerous
         cpuArch.setCPUArchitectureName(ProcessorArchitectureEnumeration.Enum
               .forString(platform));
      }

      return cpuArch;
   }

   /**
    * @see eu.unicore.hila.job.spi.NativeToModel#getFormatName()
    */
   public String getFormatName() {
      return ModelToEmiAdl.EMI_ADL;
   }

}
