/*********************************************************************************
 * Copyright (c) 2012 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.gridftp;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Vector;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import org.apache.log4j.Logger;
import org.globus.ftp.FileInfo;
import org.globus.ftp.GridFTPClient;
import org.globus.ftp.MlsxEntry;
import org.globus.ftp.exception.ClientException;
import org.globus.ftp.exception.ServerException;

import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseFile;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.exceptions.HiLANotImplementedException;
import eu.unicore.hila.exceptions.HiLANotSupportedException;
import eu.unicore.hila.grid.File;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.Storage;
import eu.unicore.hila.grid.ThirdPartyTransfer;

/**
 * {@link MlsxEntry} example:
 * unix.owner=root;unix.mode=0755;size=4096;perm=el;type
 * =dir;unix.group=root;unique=fd00-58d381;modify=20110218071710; selinux
 * 
 * @author bjoernh
 * 
 *         06.02.2012 16:22:30
 * 
 */
@ResourceType(locationStructure = {
      GridFTPImplementationProvider.SCHEME + "://{hostname}:{port}/.+?",
      GridFTPImplementationProvider.SCHEME + "://{hostname}/.+?" })
public class GridFTPFile extends BaseFile implements File {
   private static final int UNIX_READ = 4;
   private static final int UNIX_WRITE = 2;
   private static final int UNIX_EXEC = 1;

   private static final Logger log = Logger.getLogger(GridFTPFile.class);

   private MlsxEntry mlsxEntry;
   private long mlsxEntryTimestamp = 0;
   private GridFTPStorage storage;

   private static final Cache filesCache;
   static {
      CacheManager cm = CacheManager.getInstance();
      cm.addCache(new Cache(GridFTPFile.class.getName(), 500, true, true, 0, 0));
      filesCache = cm.getCache(GridFTPFile.class.getName());
   }

   /**
    * @param _location
    * @throws HiLAException
    */
   GridFTPFile(Location _location, Object... _extraInformation)
         throws HiLAException {
      super(_location);

      for (Object object : _extraInformation) {
         if (object instanceof FileInfo) {
            FileInfo fileInfo = (FileInfo) object;
         } else if (object instanceof GridFTPStorage) {
            this.storage = (GridFTPStorage) object;
         }
      }

      if (this.storage == null) {
         storage = (GridFTPStorage) findParentLocationOfType(Storage.class,
               location,
               Storage.class).locate();
         if (this.storage == null) {
            throw new HiLAFactoryException("Cannot instantiate file "
                  + this.location + " w/o storage.");
         }
      }

      this.path = storage.getLocation().relativePath(location);
   }

   public static synchronized Resource locate(Location _location,
         Object... _extraInformation) throws HiLAException {
      Element cacheElem = filesCache.get(_location);
      if (cacheElem != null) {
         return (Resource) cacheElem.getObjectValue();
      } else {
         GridFTPFile file = new GridFTPFile(_location, _extraInformation);
         filesCache.put(new Element(_location, file));
         return file;
      }
   }

   /**
    * @see eu.unicore.hila.grid.File#ls()
    */
   @Override
   public List<File> ls() throws HiLAException {

      if (!isDirectory()) {
         return Collections.singletonList((File) this);
      }

      List<File> files = new ArrayList<File>();

      try {
         final GridFTPClient client = storage.lockClient();
         client.setPassive();
         client.setLocalActive();
         final Vector rawFileList = client.list(path);
         for (Object fileInfoObj : rawFileList) {
            FileInfo fileInfo = (FileInfo) fileInfoObj;
            if (fileInfo.getName().equals("..")
                  || fileInfo.getName().equals(".")) {
               continue;
            }
            Location fileLoc = location.getChildLocation(fileInfo.getName());
            File file = (File) fileLoc.locate(fileInfo, storage);
            files.add(file);
         }
      } catch (ServerException e) {
         throw new HiLAException("Unable to list file(s) for " + location, e);
      } catch (ClientException e) {
         throw new HiLAException("Unable to list file(s) for " + location, e);
      } catch (IOException e) {
         throw new HiLAException("Unable to list file(s) for " + location, e);
      } finally {
         storage.unlockClient();
      }

      return files;

   }

   /**
    * @see eu.unicore.hila.grid.File#exists()
    */
   @Override
   public boolean exists() throws HiLAException {
      try {
         final GridFTPClient client = storage.lockClient();
         client.setPassive();
         client.setLocalActive();
         return client.exists(path);
      } catch (Exception e) {
         throw new HiLAException("Unable to determine existence of file: "
               + location, e);
      } finally {
         storage.unlockClient();
      }

   }

   /**
    * @throws HiLAException
    * 
    */
   private synchronized void updateFileInfo() throws HiLAException {
      if (mlsxEntry != null
            && (System.currentTimeMillis() < mlsxEntryTimestamp + 500)) {
         return;
      }

      // String realPath = path.isEmpty() ? "." : path;
      String parentPath = storage.getLocation().relativePath(
            location.getParentLocation());
      if (log.isDebugEnabled()) {
         log.debug("parentPath == " + parentPath);
      }

      try {
         final GridFTPClient client = storage.lockClient();
         // client.setPassive();
         // client.setLocalActive();
         mlsxEntry = client.mlst(path);
         mlsxEntryTimestamp = System.currentTimeMillis();
      } catch (Exception e) {
         throw new HiLAException("Unable to update file information.", e);
      } finally {
         storage.unlockClient();
      }

   }

   /**
    * @see eu.unicore.hila.grid.File#isReadable()
    */
   @Override
   public boolean isReadable() throws HiLAException {
      updateFileInfo();
      if (exists()) {
         int userMode = mlsxEntry.get("unix.mode").charAt(1) - '0';
         return (userMode & UNIX_READ) > 0;
      } else {
         throw new HiLAException("File does not exist: " + location);
      }
   }

   /**
    * @see eu.unicore.hila.grid.File#setIsReadable(boolean)
    */
   @Override
   public void setIsReadable(boolean _readable) throws HiLAException {
      throw new HiLANotSupportedException(
            "GridFTP does not support setting permissions.");
   }

   /**
    * @see eu.unicore.hila.grid.File#isWritable()
    */
   @Override
   public boolean isWritable() throws HiLAException {
      updateFileInfo();
      if (exists()) {
         int userMode = mlsxEntry.get("unix.mode").charAt(1) - '0';
         return (userMode & UNIX_WRITE) > 0;
      } else {
         throw new HiLAException("File does not exist: " + location);
      }
   }

   /**
    * @see eu.unicore.hila.grid.File#setIsWritable(boolean)
    */
   @Override
   public void setIsWritable(boolean _writable) throws HiLAException {
      throw new HiLANotSupportedException(
            "GridFTP does not support setting permissions.");
   }

   /**
    * @see eu.unicore.hila.grid.File#isExecutable()
    */
   @Override
   public boolean isExecutable() throws HiLAException {
      updateFileInfo();
      if (exists()) {
         int userMode = mlsxEntry.get("unix.mode").charAt(1) - '0';
         return (userMode & UNIX_EXEC) > 0;
      } else {
         throw new HiLAException("File does not exist: " + location);
      }
   }

   /**
    * @see eu.unicore.hila.grid.File#setIsExecutable(boolean)
    */
   @Override
   public void setIsExecutable(boolean _executable) throws HiLAException {
      throw new HiLANotSupportedException(
            "GridFTP does not support setting permissions.");
   }

   /**
    * @see eu.unicore.hila.grid.File#delete(boolean)
    */
   @Override
   public boolean delete(boolean _recursive) throws HiLAException {
      try {
         final GridFTPClient client = storage.lockClient();
         client.setPassive();
         client.setLocalActive();
         if (exists() && isDirectory() && _recursive) {
            client.deleteDir(path);
            return true;
         } else if (exists() && !isDirectory()) {
            client.deleteFile(path);
            return true;
         } else if (!exists()) {
            return false;
         }
      } catch (Exception e) {
         throw new HiLAException("Unable to delete " + location, e);
      } finally {
         storage.unlockClient();
      }

      return false;
   }

   /**
    * @see eu.unicore.hila.grid.File#mkdir(boolean)
    */
   @Override
   public boolean mkdir(boolean _createParents) throws HiLAException {
      try {
         final GridFTPClient client = storage.lockClient();
         try {
            client.makeDir(path);
            return true;
         } catch (Exception e) {
            throw new HiLAException("Unable to create directory " + location, e);
         }
      } finally {
         storage.unlockClient();
      }
   }

   /**
    * @see eu.unicore.hila.grid.File#getStorage()
    */
   @Override
   public Storage getStorage() throws HiLAException {
      return storage;
   }

   /**
    * @see eu.unicore.hila.grid.File#importFromLocalFile(java.io.File, boolean, boolean)
    */
   @Override
   public SimpleTransfer importFromLocalFile(java.io.File localFile,
         boolean _overwrite, boolean _recursive) throws HiLAException {
      try {
         return importFromStream(new FileInputStream(localFile), _overwrite);
      } catch (FileNotFoundException e) {
         throw new HiLAException("File not found.", e);
      }
   }

   /**
    * @see eu.unicore.hila.grid.File#exportToLocalFile(java.io.File, boolean, boolean)
    */
   @Override
   public SimpleTransfer exportToLocalFile(java.io.File localFile,
         boolean _overwrite, boolean _recursive) throws HiLAException {
      java.io.File actualFile = localFile;
      if (localFile.exists() && localFile.isDirectory()) {
         actualFile = new java.io.File(localFile, getName());
      }
      if (!actualFile.exists() || _overwrite) {
         try {
            FileOutputStream fos = new FileOutputStream(actualFile);
            return exportToStream(fos, true);
         } catch (FileNotFoundException e) {
            throw new HiLAException("File not found.", e);
         }

      } else {
         throw new HiLAException(
               "Local file exists, but must not be overwritten.");
      }

   }

   /**
    * Default implementation does not close the output stream.
    * 
    * @see eu.unicore.hila.grid.File#exportToStream(java.io.OutputStream)
    */
   @Override
   public SimpleTransfer exportToStream(OutputStream _os) throws HiLAException {
      return exportToStream(_os, false);
   }

   /**
    * @param _closeOutputStream
    * @see eu.unicore.hila.grid.File#exportToStream(java.io.OutputStream)
    */
   private SimpleTransfer exportToStream(OutputStream _os,
         boolean _closeOutputStream)
         throws HiLAException {
      return new GridFtpExport(storage, path, _os, _closeOutputStream);
   }

   /**
    * @see eu.unicore.hila.grid.File#importFromStream(java.io.InputStream, boolean)
    */
   @Override
   public SimpleTransfer importFromStream(InputStream _is, boolean _overwrite)
         throws HiLAException {
      if (!exists() || _overwrite) {
         return new GridFtpImport(storage, path, _is, false);
      } else {
         throw new HiLAException("File exists, but cannot be overwritten.");
      }
   }

   /**
    * @see eu.unicore.hila.grid.File#transfer(eu.unicore.hila.grid.File, boolean, boolean)
    */
   @Override
   public ThirdPartyTransfer transfer(File _other, boolean _overwrite,
         boolean _recursive) throws HiLAException {
      throw new HiLANotImplementedException("");
   }

   /**
    * @see eu.unicore.hila.grid.File#size()
    */
   @Override
   public long size() throws HiLAException {
      updateFileInfo();
      return Long.parseLong(mlsxEntry.get(MlsxEntry.SIZE));
   }

   /**
    * @see eu.unicore.hila.grid.File#lastModified()
    */
   @Override
   public Date lastModified() throws HiLAException {

      final GridFTPClient client = storage.lockClient();
      try {
         client.setPassive();
         client.setLocalActive();
         return client.lastModified(path);
      } catch (Exception e) {
         throw new HiLAException("Unable to determine last modified date for "
               + location, e);
      } finally {
         storage.unlockClient();
      }

   }

   /**
    * @see eu.unicore.hila.grid.File#copyTo(eu.unicore.hila.grid.File, boolean, boolean)
    */
   @Override
   public void copyTo(File _other, boolean _overwrite, boolean _recursive)
         throws HiLAException {
      throw new HiLANotSupportedException(
            "GridFTP does not support local file copies.");
   }

   /**
    * @see eu.unicore.hila.grid.File#moveTo(eu.unicore.hila.grid.File, boolean)
    */
   @Override
   public void moveTo(File _other, boolean _overwrite) throws HiLAException {
      File realFile = _other;
      if (_other.exists() && _other.isDirectory()) {
         realFile = (File) _other.getChild(getName());
      }

      if (realFile.exists() || _overwrite) {

         try {
            final GridFTPClient client = storage.lockClient();
            client.setPassive();
            client.setLocalActive();
            client.rename(path, ((GridFTPFile) realFile).path);
         } catch (Exception e) {
            throw new HiLAException("Unable to rename file.", e);
         } finally {
            storage.unlockClient();
         }
      } else {
         throw new HiLAException("moveTo(): target exists");
      }

   }

   /**
    * @see eu.unicore.hila.grid.File#isDirectory()
    */
   
   @Override
   public boolean isDirectory() throws HiLAException {
      if (mlsxEntry != null) {
         return mlsxEntry.get(MlsxEntry.TYPE).equals(MlsxEntry.TYPE_DIR);
      } else {
         try {
            GridFTPClient client = storage.lockClient();
            try {
               // client.setPassive();
               // client.setLocalActive();
               MlsxEntry mlsxEntry = client.mlst(path);
               return mlsxEntry.get(MlsxEntry.TYPE).equals(MlsxEntry.TYPE_DIR);
            } catch (Exception e) {
               throw new HiLAException(
                     "Unable to determine whether this is a directory.", e);
            }
         } finally {
            storage.unlockClient();
         }
      }
   }
    


}
