/*********************************************************************************
 * Copyright (c) 2012 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.emi.es;

import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

import javax.ws.rs.core.MultivaluedMap;

import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONObject;

import com.sun.jersey.core.util.MultivaluedMapImpl;

import eu.emi.emir.client.EMIRClient;
import eu.emi.emir.client.ServiceBasicAttributeNames;
import eu.unicore.emi.es.clients.ActivityInfoClient;
import eu.unicore.emi.es.clients.ActivityManagementClient;
import eu.unicore.emi.es.clients.CreateActivityClient;
import eu.unicore.hila.Location;
import eu.unicore.hila.grid.Site;
import eu.unicore.util.httpclient.ClientProperties;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * This {@link Callable} resolves {@link Site}s from an EMIR registry given it
 * {@link EMIRClient}. It uses the following strategy:
 * 
 * <ol>
 * <li>Look for service endpoint interface name "CreateActivityService"</li>
 * <li>For all entries returned, find "ActivityInfoService" and
 * "ActivityManagementService" with the same service Id as
 * "CreateActivityService"</li>
 * <li>Combine matching services (same service Id) to a {@link Site}</li>
 * <ol>
 * 
 * @author bjoernh
 * 
 *         12.07.2012 09:22:58
 * 
 */
public class EmirSiteDiscovery implements Callable<List<Site>> {

	private final EMIRClient emirClient;
	private EmiEsProperties props;

	/**
	 * @param _client
	 * @param props
	 * 
	 */
	public EmirSiteDiscovery(EMIRClient _client, EmiEsProperties _props) {
		emirClient = _client;
		this.props = _props;
	}

	/**
	 * @see java.util.concurrent.Callable#call()
	 */
	@Override
	public List<Site> call() throws Exception {
		final List<Site> sites = new ArrayList<Site>();

		final MultivaluedMap<String, String> params1 = new MultivaluedMapImpl();

		params1.add(ServiceBasicAttributeNames.SERVICE_ENDPOINT_IFACENAME
				.getAttributeName(), "org.ogf.glue.emies.activitycreation");

		final JSONArray createActivityEntries = emirClient
				.queryByQueryParams(params1);

		Map<String, SiteServices> sitesServices = new HashMap<String, SiteServices>();
		// the last entry is always the ref
		for (int i = 0; i < createActivityEntries.length() - 1; i++) {
			JSONObject createActivityEntry = createActivityEntries
					.getJSONObject(i);
			String serviceId = createActivityEntry
					.getString(ServiceBasicAttributeNames.SERVICE_ID
							.getAttributeName());
			String createActivityServiceUrl = createActivityEntry
					.getString(ServiceBasicAttributeNames.SERVICE_ENDPOINT_URL
							.getAttributeName());

			final MultivaluedMap<String, String> params2 = new MultivaluedMapImpl();
			params2.add(
					ServiceBasicAttributeNames.SERVICE_ID.getAttributeName(),
					serviceId);
			params2.add(ServiceBasicAttributeNames.SERVICE_ENDPOINT_IFACENAME
					.getAttributeName(),
					"org.ogf.glue.emies.activitymanagement");
			final JSONArray activityMgmtEntries = emirClient
					.queryByQueryParams(params2);
			String amURL = activityMgmtEntries.getJSONObject(0).getString(
					ServiceBasicAttributeNames.SERVICE_ENDPOINT_URL
							.getAttributeName());

			final MultivaluedMap<String, String> params3 = new MultivaluedMapImpl();
			params3.add(
					ServiceBasicAttributeNames.SERVICE_ID.getAttributeName(),
					serviceId);
			params3.add(ServiceBasicAttributeNames.SERVICE_ENDPOINT_IFACENAME
					.getAttributeName(), "org.ogf.glue.emies.activityinfo");
			final JSONArray activityInfoEntries = emirClient
					.queryByQueryParams(params3);
			String aiURL = activityInfoEntries.getJSONObject(0).getString(
					ServiceBasicAttributeNames.SERVICE_ENDPOINT_URL
							.getAttributeName());

			final SiteServices singleSiteServices = new SiteServices();
			singleSiteServices
					.setCreateActivityService(createActivityServiceUrl);
			singleSiteServices.setActivityInfoService(aiURL);
			singleSiteServices.setActivityMgmtService(amURL);
			sitesServices.put(serviceId, singleSiteServices);

			// we use the creation service URL as Site URL to generate its name,
			// which will simply be the hostname
			final URL siteURL = new URL(createActivityServiceUrl);
			final String siteName = siteURL.getHost();

			Location siteLoc = new Location(EmiEsImplementationProvider.SCHEME
					+ ":/sites").getChildLocation(siteName);
			IClientConfiguration sp = new ClientProperties(props);
			ActivityInfoClient aiClient = new ActivityInfoClient(aiURL, sp);
			ActivityManagementClient amClient = new ActivityManagementClient(
					amURL, sp);
			CreateActivityClient caClient = new CreateActivityClient(
					createActivityServiceUrl, sp);

			Site site = (Site) siteLoc.locate(new Object[] { aiClient,
					amClient, caClient });
			sites.add(site);
		}

		return sites;
	}

}
