/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.emi.es;

import java.io.File;
import java.util.Collections;
import java.util.List;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import org.apache.log4j.Logger;

import eu.emi.es.x2010.x12.activity.ActivityInfoItemDocument;
import eu.emi.es.x2010.x12.activity.ActivityInfoItemDocument.ActivityInfoItem;
import eu.emi.es.x2010.x12.activity.ActivityStatusItemDocument.ActivityStatusItem;
import eu.emi.es.x2010.x12.types.ActivityStatusAttribute;
import eu.emi.es.x2010.x12.types.ActivityStatusDocument.ActivityStatus;
import eu.emi.es.x2010.x12.types.ActivityStatusState;
import eu.unicore.emi.es.clients.ActivityInfoClient;
import eu.unicore.emi.es.clients.ActivityManagementClient;
import eu.unicore.emi.es.faults.AccessControlFault;
import eu.unicore.emi.es.faults.InternalBaseFault;
import eu.unicore.emi.es.faults.UnknownAttributeException;
import eu.unicore.emi.es.faults.VectorLimitExceededFault;
import eu.unicore.hila.Location;
import eu.unicore.hila.Metadata;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseJob;
import eu.unicore.hila.exceptions.HiLAAccessDeniedException;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.TaskStatus;

/**
 * @author bjoernh
 * 
 *         21.12.2011 10:22:34
 * 
 */
@ResourceType(locationStructure = {
		EmiEsImplementationProvider.SCHEME
				+ ":/sites/{siteName}/tasks/{taskId}/?",
		EmiEsImplementationProvider.SCHEME
				+ ":/{user}@sites/{siteName}/tasks/{taskId}/?" })
public class EmiEsJob extends BaseJob {

	private static final Logger log = Logger.getLogger(EmiEsJob.class);

	private ActivityManagementClient actMgmtClient = null;
	private ActivityInfoClient actInfoClient = null;

	private static final Cache tasksCache;
	static {
		final CacheManager manager = CacheManager.getInstance();
		manager.addCache(new Cache(EmiEsJob.class.getName(), 10000, true, true,
				0, 0));
		tasksCache = manager.getCache(EmiEsJob.class.getName());
	}

	/**
	 * @param location
	 * @param _extraInformation
	 * @throws HiLAFactoryException
	 */
	EmiEsJob(Location location, Object... _extraInformation)
			throws HiLAFactoryException {
		super(location);
		for (Object object : _extraInformation) {
			if (object instanceof ActivityManagementClient) {
				actMgmtClient = (ActivityManagementClient) object;
			} else if (object instanceof ActivityInfoClient) {
				actInfoClient = (ActivityInfoClient) object;
			}
		}
		if (actMgmtClient == null || actInfoClient == null) {
			throw new HiLAFactoryException("Unable to create job object.");
		}
		setCurrentState(TaskStatus.NEW);

		synchronized (tasksCache) {
			tasksCache.put(new Element(location, this));
		}

	}

	public static final Resource locate(Location _location,
			Object... _extraInformation) throws HiLAException {
		if (tasksCache.isKeyInCache(_location)) {
			Element cacheElem = tasksCache.get(_location);
			if (cacheElem != null) {
				return (Resource) cacheElem.getObjectValue();
			}
		}

		return new EmiEsJob(_location, _extraInformation);

		// return ((EmiEsSite) findParentLocationOfType(EmiEsSite.class,
		// _location,
		// EmiEsGrid.class).locate()).getTask(_location.getName());
	}

	/**
	 * @see eu.unicore.hila.grid.Job#startASync(java.io.File[])
	 */
	@Override
	public void startASync(File... imports) throws HiLAException {
		// TODO put files into working directory
		try {
			actMgmtClient.dataPushDone(getId());
		} catch (Exception e) {
			throw new HiLAException("Unable to signal data push done.", e);
		}
	}

	/**
	 * @see eu.unicore.hila.grid.Job#cleanup(eu.unicore.hila.grid.File[])
	 */
	@Override
	public List<SimpleTransfer> cleanup(eu.unicore.hila.grid.File... exports)
			throws HiLAException {
		// TODO Incomplete implementation just wiping the activity, need to also
		// export files. Should we throw an exception?
		try {
			actMgmtClient.wipeActivity(getName());
		} catch (InternalBaseFault e) {
			throw new HiLAException("Unable to wipe activity " + getName(), e);
		} catch (VectorLimitExceededFault e) {
			throw new HiLAException("Unable to wipe activity " + getName(), e);
		} catch (AccessControlFault e) {
			throw new HiLAException("Unable to wipe activity " + getName(), e);
		}
		return Collections.emptyList();
	}

	/**
	 * @see eu.unicore.hila.grid.Job#getExitCode()
	 */
	@Override
	public int getExitCode() throws HiLAException {
		try {
			return actMgmtClient.getActivityInfo(getId()).get(0)
					.getActivityInfoDocument().getExitCode();
		} catch (InternalBaseFault e) {
			throw new HiLAException("Unable to retrieve exit code of activity "
					+ getId(), e);
		} catch (VectorLimitExceededFault e) {
			throw new HiLAException("Unable to retrieve exit code of activity "
					+ getId(), e);
		} catch (UnknownAttributeException e) {
			throw new HiLAException("Unable to retrieve exit code of activity "
					+ getId(), e);
		}
	}

	/**
	 * @see eu.unicore.hila.grid.Job#getLog()
	 */
	@Override
	public String getLog() throws HiLAException {
		// TODO Auto-generated method stub
		try {
			List<ActivityInfoItem> actInfo = actMgmtClient
					.getActivityInfo(getId());
			if (actInfo.size() != 1) {
				throw new HiLAException(
						"Expected exactly one information item, got "
								+ actInfo.size());
			}
			// TODO this is just returning a reference to the log, not the log
			// itself
			return actInfo.get(0).getActivityInfoDocument().getLogDir();
		} catch (InternalBaseFault e) {
			throw new HiLAException("Unable to retrieve job log.", e);
		} catch (VectorLimitExceededFault e) {
			throw new HiLAException("Requested too many jobs' information", e);
		} catch (UnknownAttributeException e) {
			throw new HiLAException("Unknown activity.", e);
		}
	}

	/**
	 * @see eu.unicore.hila.common.grid.BaseJob#getMetadata()
	 */
	@Override
	public Metadata getMetadata() throws HiLAException {
		Metadata md = new Metadata();
		try {
			List<ActivityInfoItem> ainfos = actMgmtClient
					.getActivityInfo(getId());
			if (ainfos.size() > 0) {
				ActivityInfoItem ainfo = ainfos.get(0);
				ActivityInfoItemDocument aid = ActivityInfoItemDocument.Factory
						.newInstance();
				aid.setActivityInfoItem(ainfo);

				md.setData("activityInfo", aid.xmlText());
				return md;
			}
			throw new HiLAException(
					"Unable to retrieve activity info. Empty list from server.");
		} catch (InternalBaseFault e) {
			throw new HiLAException("Unable to retrieve activity info.", e);
		} catch (VectorLimitExceededFault e) {
			throw new HiLAException("Unable to retrieve activity info.", e);
		} catch (UnknownAttributeException e) {
			throw new HiLAException("Unable to retrieve activity info.", e);
		}
	}

	/**
	 * @see eu.unicore.hila.grid.Job#getStdErr()
	 */
	@Override
	public eu.unicore.hila.grid.File getStdErr() throws HiLAException {
		String stdErrEpr;
		try {
			final ActivityInfoItem aInfo = actMgmtClient.getActivityInfo(
					getId()).get(0);
			final String stdErrName = aInfo.getActivityInfoDocument()
					.getStdErr();
			return (eu.unicore.hila.grid.File) getOutputDir(aInfo).getChild(
					stdErrName == null ? "stderr" : stdErrName);
		} catch (InternalBaseFault e) {
			throw new HiLAException("Unable to get standard out for job "
					+ getLocation(), e);
		} catch (VectorLimitExceededFault e) {
			throw new HiLAException("Unable to get standard out for job "
					+ getLocation(), e);
		} catch (UnknownAttributeException e) {
			throw new HiLAException("Unable to get standard out for job "
					+ getLocation(), e);
		} catch (Exception e) {
			throw new HiLAException("Unable to get standard out for job "
					+ getLocation());
		}
	}

	/**
	 * @see eu.unicore.hila.grid.Job#getStdOut()
	 */
	@Override
	public eu.unicore.hila.grid.File getStdOut() throws HiLAException {

		try {
			final ActivityInfoItem aInfo = actMgmtClient.getActivityInfo(
					getId()).get(0);
			final String stdOutName = aInfo.getActivityInfoDocument()
					.getStdOut();
			return (eu.unicore.hila.grid.File) getOutputDir(aInfo).getChild(
					stdOutName == null ? "stdout" : stdOutName);
		} catch (InternalBaseFault e) {
			throw new HiLAException("Unable to get standard out for job "
					+ getLocation(), e);
		} catch (VectorLimitExceededFault e) {
			throw new HiLAException("Unable to get standard out for job "
					+ getLocation(), e);
		} catch (UnknownAttributeException e) {
			throw new HiLAException("Unable to get standard out for job "
					+ getLocation(), e);
		}

	}

	private eu.unicore.hila.grid.File getSessionDir(ActivityInfoItem aInfo)
			throws HiLAException {

		final String[] sessDirUrl = aInfo.getActivityInfoDocument()
				.getSessionDirectoryArray();
		for (int i = 0; i < sessDirUrl.length; i++) {
			try {
				return (eu.unicore.hila.grid.File) new Location(sessDirUrl[i])
						.locate(actInfoClient.getSecurityProperties());
			} catch (HiLAException e) {
				// ignore, we're searching a usable protocol here
				// debug logging is a good idea
				if (log.isDebugEnabled()) {
					log.debug("No implementation of "
							+ eu.unicore.hila.grid.File.class.getName()
							+ " for " + sessDirUrl[i]);
				}
			}
		}
		throw new HiLAException(
				"Cannot find usable protocol for session directory.");
	}

	private eu.unicore.hila.grid.File getOutputDir(ActivityInfoItem _aInfo)
			throws HiLAException {
		final String[] outputDirUrls = _aInfo.getActivityInfoDocument()
				.getStageOutDirectoryArray();
		for (String outputDirUrl : outputDirUrls) {

			try {
				return (eu.unicore.hila.grid.File) new Location(outputDirUrl)
						.locate(actInfoClient.getSecurityProperties());
			} catch (HiLAException e) {
				// ignore, we're searching a usable protocol here
				// debug logging is a good idea
				if (log.isDebugEnabled()) {
					log.debug("No implementation of "
							+ eu.unicore.hila.grid.File.class.getName()
							+ " for " + outputDirUrl);
				}

			}
		}
		throw new HiLAException(
				"Cannot find usable protocol for output directory.");
	}

	/**
	 * @see eu.unicore.hila.grid.Job#getWorkingDirectory()
	 */
	@Override
	public eu.unicore.hila.grid.File getWorkingDirectory() throws HiLAException {
		ActivityInfoItem aInfo;
		try {
			aInfo = actMgmtClient.getActivityInfo(getId()).get(0);
		} catch (InternalBaseFault e) {
			throw new HiLAException("Unable to get properties for job "
					+ getLocation(), e);
		} catch (VectorLimitExceededFault e) {
			throw new HiLAException("Unable to get properties for job "
					+ getLocation(), e);
		} catch (UnknownAttributeException e) {
			throw new HiLAException("Unable to get properties for job "
					+ getLocation(), e);
		}
		return getSessionDir(aInfo);

	}

	/**
	 * @see eu.unicore.hila.grid.Task#abort()
	 */
	@Override
	public void abort() throws HiLAException {
		try {
			actMgmtClient.cancelActivity(getId());
		} catch (InternalBaseFault e) {
			throw new HiLAException("Unable to abort job.", e);
		} catch (VectorLimitExceededFault e) {
			throw new HiLAException("Unable to abort job.", e);
		} catch (AccessControlFault e) {
			throw new HiLAException("Unable to abort job.", e);
		}
	}

	/**
	 * @see eu.unicore.hila.grid.Task#getId()
	 */
	@Override
	public String getId() throws HiLAException {
		return getName();
	}

	/**
	 * @see eu.unicore.hila.common.grid.BaseTask#status()
	 */
	@Override
	public TaskStatus status() throws HiLAException {
		if (super.status().isFinalState()) {
			return super.status();
		} else {
			return updateStatus();
		}
	}

	/**
	 * @return
	 * @throws HiLAException
	 */
	private TaskStatus updateStatus() throws HiLAException {
		try {
			if (log.isDebugEnabled()) {
				log.debug("Updating status of " + location);
			}
			List<ActivityStatusItem> actStatusResp = actMgmtClient
					.getActivityStatus(getId());
			TaskStatus newStatus = mapStatus(actStatusResp.get(0)
					.getActivityStatus());
			setCurrentState(newStatus);
			return newStatus;
		} catch (InternalBaseFault e) {
			throw new HiLAException("", e);
		} catch (VectorLimitExceededFault e) {
			throw new HiLAException("", e);
		} catch (AccessControlFault e) {
			throw new HiLAAccessDeniedException(e.getMessage(), e);
		} catch (HiLAException e) {
			throw new HiLAException("Unable to determine job id.", e);
		}
	}

	/**
	 * @param activityStatus
	 * @return
	 * @throws HiLAException
	 */
	private TaskStatus mapStatus(ActivityStatus activityStatus)
			throws HiLAException {
		switch (activityStatus.getStatus().intValue()) {
		// TODO check for the various PROCESSING statuses
		case (ActivityStatusState.INT_ACCEPTED):
			return TaskStatus.NEW;
		case (ActivityStatusState.INT_POSTPROCESSING):
		case (ActivityStatusState.INT_PROCESSING):
		case (ActivityStatusState.INT_PROCESSING_RUNNING):
			return TaskStatus.RUNNING;
		case (ActivityStatusState.INT_PREPROCESSING):
		case (ActivityStatusState.INT_PROCESSING_ACCEPTING):
		case (ActivityStatusState.INT_PROCESSING_QUEUED):
			return TaskStatus.PENDING;
		case (ActivityStatusState.INT_TERMINAL):
			for (ActivityStatusAttribute.Enum attribute : activityStatus
					.getAttributeArray()) {
				switch (attribute.intValue()) {
				case ActivityStatusAttribute.INT_APP_FAILURE:
				case ActivityStatusAttribute.INT_EXPIRED:
				case ActivityStatusAttribute.INT_POSTPROCESSING_FAILURE:
				case ActivityStatusAttribute.INT_PROCESSING_FAILURE:
					return TaskStatus.FAILED;
				case ActivityStatusAttribute.INT_POSTPROCESSING_CANCEL:
				case ActivityStatusAttribute.INT_PROCESSING_CANCEL:
					return TaskStatus.ABORTED;
				default:
					break;
				}
			}
			// if not failed or aborted, it must have benn successful
			return TaskStatus.SUCCESSFUL;
		}
		throw new HiLAException("Cannot map status "
				+ activityStatus.getStatus().toString()
				+ " to any known HiLA status.");
	}
}
