/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.common;

import eu.unicore.hila.Location;
import eu.unicore.hila.Metadata;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationPatternException;
import eu.unicore.hila.exceptions.HiLANotImplementedException;

/**
 * @author bjoernh
 * 
 *         13.08.2009 10:49:26
 * 
 */
public abstract class BaseResource implements Resource {

	final protected Location location;

	/**
     * 
     */
	public BaseResource(Location _location) {
		this.location = _location;
	}

	/**
	 * @see eu.unicore.hila.Resource#getLocation()
	 */
	public final Location getLocation() {
		return this.location;
	}

	/**
	 * @throws HiLAException
	 * @see eu.unicore.hila.Resource#getChild(java.lang.String)
	 */
	public Resource getChild(String childPath) throws HiLAException {
		Location childLocation = this.location.getChildLocation(childPath);
		return childLocation.locate();
	}

	/**
	 * @see eu.unicore.hila.Resource#getMetadata()
	 */
	@Override
	public Metadata getMetadata() throws HiLAException {
		throw new HiLANotImplementedException("Not implemented");
	}

	/**
	 * @see eu.unicore.hila.Resource#getName()
	 */
	public String getName() {
		return location.getName();
	}

	/**
	 * @throws HiLAException
	 * @see eu.unicore.hila.Resource#getParent()
	 */
	public Resource getParent() throws HiLAException {
		return location.getParentLocation().locate();
	}

	/**
	 * Extract a {@link Location} of type _class from the deeper _location, e.g.
	 * the parent Site Location of a Task (in the Grid API hila-grid-api).
	 * 
	 * @param _child
	 *            {@link Location} of the {@link Resource}.
	 * @param _type
	 *            the {@link ResourceType} we're looking for.
	 * @param _stopAtType
	 *            stop search at this type, if it cannot be expected that a
	 *            resource of type _class can be found as a parent of _child
	 * @return The {@link Storage} location.
	 * @throws HiLAException
	 */
	public static Location findParentLocationOfType(
			Class<? extends Resource> _type, Location _child,
			Class<? extends Resource> _stopAtType)
			throws HiLALocationPatternException {
		Location currentLocation = _child;
		while (!(currentLocation.isLocationOfType(_type) || currentLocation
				.isLocationOfType(_stopAtType))) {
			currentLocation = currentLocation.getParentLocation();
		}
		if (currentLocation.isLocationOfType(_type)) {
			return currentLocation;
		} else {
			throw new HiLALocationPatternException("No Location of type "
					+ _type + " is parent of: " + _child);
		}
	}

	/**
	 * @see eu.unicore.hila.Resource#ok()
	 */
	@Override
	public boolean ok() {
		return true;
	}
}
