/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Properties;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;

import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;

public class Location {

	static {
		Properties log4jProps = new Properties();
		try {
			final File log4jConf = new File("log4j.properties");
			if (log4jConf.exists()) {
				InputStream is = new FileInputStream(log4jConf);
				log4jProps.load(is);
				is.close();
			} else {
				InputStream is = Location.class
						.getResourceAsStream("/log4j.properties");
				log4jProps.load(is);
				is.close();
			}
			PropertyConfigurator.configure(log4jProps);
		} catch (IOException e) {
			System.out.println("Unable to setup log4j.");
		}
	}

	private static final Logger log = Logger.getLogger(Location.class);

	private final URI uri;

	private volatile Location parentLocation = null;

	/**
	 * @throws HiLALocationSyntaxException
	 * 
	 */
	public Location(String stringLocation) throws HiLALocationSyntaxException {
		try {
			this.uri = new URI(stringLocation);
			this.uri.normalize();
		} catch (URISyntaxException e) {
			throw new HiLALocationSyntaxException("Invalid HiLA Location.", e);
		}
	}

	public Location(URI uri) {
		this.uri = uri;
		this.uri.normalize();
	}

	public Location getParentLocation() {
		if (parentLocation == null) {
			synchronized (this) {
				if (parentLocation == null) {
					if (this.getStringValue().charAt(
							this.getStringValue().length() - 1) == '/') {
						parentLocation = new Location(uri.resolve(".."));
					} else {
						parentLocation = new Location(uri.resolve("."));
					}
				}
			}
		}
		return parentLocation;
	}

	public Class<Resource> getLocationType() {
		return ResourceTypeRegistry.getInstance().determineResourceType(this);
	}

	public boolean isLocationOfType(Class type) {
		Class<Resource> actualType = ResourceTypeRegistry.getInstance()
				.determineResourceType(this);
		if (actualType != null) {
			if (actualType.equals(type)) {
				return true;
			} else {
				return implementsOrExtends(type, actualType);
			}
		} else {
			return false;
		}
	}

	public boolean isAbsolute() {
		return uri.isAbsolute();
	}

	/**
	 * @param type
	 *            The type that should be extended or implemented
	 *            (supertype/interface)
	 * @param actualType
	 *            The actual type (subtype) of this location.
	 * @return
	 */
	private boolean implementsOrExtends(Class type, Class actualType) {
		if (log.isTraceEnabled()) {
			log.trace("Checking whether " + actualType.getName()
					+ " implements or extends " + type.getName());
		}
		if (type.equals(actualType)) {
			return true;
		}
		Class<?>[] interfaces = actualType.getInterfaces();
		for (Class<?> class1 : interfaces) {
			if (type.equals(class1)) {
				return true;
			}
		}
		Class superClass = actualType.getSuperclass();
		if (superClass != null) {
			return implementsOrExtends(type, superClass);
		} else {
			return false;
		}

	}

	public Resource locate(Object... extraInformation) throws HiLAException {
		Class<Resource> resourceType = ResourceTypeRegistry.getInstance()
				.determineResourceType(this);
		// .getParentLocation().getChildLocation(
		// this.getName()));

		if (resourceType == null) {
         throw new HiLALocationSyntaxException("Unknown resource type.");
		}

		try {
			// this is for ResourceTypes that support caching and "know" how
			// to construct objects of their own type or simply return
			// equivalent
			// existing ones
			try {
				final Method locateMethod = resourceType.getMethod("locate",
						Location.class, Array.newInstance(Object.class, 0)
								.getClass());
				final Resource resource = (Resource) locateMethod.invoke(null,
						this, extraInformation);
				return resource;
			} catch (NoSuchMethodException e) {
				if (log.isTraceEnabled()) {
					log.trace("ResourceType " + resourceType
							+ " does not provide a locate/2 method.");
				}
				try {
					final Method locateMethod = resourceType.getMethod(
							"locate", Location.class);
					final Resource resource = (Resource) locateMethod.invoke(
							null, this);
					return resource;
				} catch (NoSuchMethodException e1) {
					if (log.isTraceEnabled()) {
						log.trace("ResourceType " + resourceType
								+ " does not provide a locate/1 method.");
					}
				} catch (SecurityException e1) {
					throw new HiLAException(
							"Access to resource's locate method denied.", e1);
				} catch (IllegalArgumentException e1) {
					throw new HiLAException(
							"Illegal argument in call to locate method.", e1);
				} catch (IllegalAccessException e1) {
					throw new HiLAException(
							"Access to resource's locate method denied.", e1);
				} catch (InvocationTargetException e1) {
					if (e1.getCause() instanceof HiLAException) {
						throw (HiLAException) e1.getCause();
					} else {
						throw new HiLAFactoryException(
								"Resource constructor threw an exception.",
								e1.getCause());
					}
				}

			}

			try {
				Constructor<Resource> resourceConstructor = resourceType
						.getConstructor(Location.class);
				return resourceConstructor.newInstance(this);
			} catch (NoSuchMethodException e) {
				throw new HiLAFactoryException(
						"Unable to instantiate resource without further information.",
						e);
			} catch (InstantiationException e) {
				throw new HiLAException(
						"Unable to instantiate resource without further information.",
						e);
			}

		} catch (SecurityException e) {
			throw new HiLAException("Access to resource's constructor denied.",
					e);
		} catch (IllegalArgumentException e) {
			throw new HiLAException(
					"Illegal argument in call to resource's constructor.", e);
		} catch (IllegalAccessException e) {
			throw new HiLAException("Access to resource's constructor denied.",
					e);
		} catch (InvocationTargetException e) {
			if (e.getCause() instanceof HiLAException) {
				throw (HiLAException) e.getCause();
			} else {
				throw new HiLAFactoryException(
						"Resource constructor threw an exception.", e);
			}
		}

	}

	/**
	 * @return
	 */
	public CharSequence getStringValue() {
		return this.toString();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return this.uri.toASCIIString();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj instanceof Location) {
			Location other = (Location) obj;
			if (other.toString().endsWith("/")) {
				try {
					other = other.getParentLocation().getChildLocation(
							other.getName());
				} catch (HiLALocationSyntaxException e) {
					return false;
				}
			}
			Location thisAlt = this;
			if (this.toString().endsWith("/")) {
				try {
					thisAlt = this.getParentLocation().getChildLocation(
							this.getName());
				} catch (HiLALocationSyntaxException e) {
					return false;
				}
			}
			return thisAlt.uri.equals(other.uri);
		}
		return false;
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		if (this.toString().endsWith("/")) {
			Location loc2;
			try {
				loc2 = this.getParentLocation()
						.getChildLocation(this.getName());
				return loc2.uri.hashCode();
			} catch (HiLALocationSyntaxException e) {
				return uri.hashCode() + 1;
			}

		}
		return uri.hashCode();
	}

	/**
	 * @param childPath
	 * @return
	 * @throws HiLALocationSyntaxException
	 */
	public Location getChildLocation(String childPath)
			throws HiLALocationSyntaxException {
		if (uri.toString().endsWith("/")) {
			return new Location(uri.toString().concat(childPath));
		} else {
			return new Location(uri.toString().concat("/".concat(childPath)));
		}
	}

	public String getName() {
		String path = uri.getPath();
		if (path.endsWith("/")) {
			path = path.substring(0, path.length() - 1);
		}
		StringTokenizer tok = new StringTokenizer(path, "/");
		String name = null;
		while (tok.hasMoreTokens()) {
			name = tok.nextToken();
		}
		return (name == null) ? "" : name;
	}

	public String relativePath(Location _other) {
		URI relative = this.uri.relativize(_other.uri);
		return relative.toString();
	}

	/**
	 * @return
	 */
	public String getUser() {
		LocationPattern lp = ResourceTypeRegistry.getInstance()
				.determineLocationPattern(this);
		if (lp != null) {
			return lp.fillTemplates(this).get("{user}");
		}
		return null;
	}
}
