/*
 * Copyright (c) Members of the EMI Collaboration. 2010-2012.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0 
 */
package org.glite.sts.authn;

import java.util.List;

import java.security.cert.X509Certificate;

import javax.security.auth.x500.X500Principal;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.shibboleth.ext.spring.webflow.Event;
import net.shibboleth.ext.spring.webflow.Events;
import net.shibboleth.idp.authn.AbstractAuthenticationAction;
import net.shibboleth.idp.authn.AuthenticationException;
import net.shibboleth.idp.authn.AuthenticationRequestContext;
import net.shibboleth.idp.profile.ActionSupport;
import net.shibboleth.idp.profile.EventIds;
import net.shibboleth.idp.profile.ProfileRequestContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.webflow.execution.RequestContext;

import eu.emi.security.authn.x509.NamespaceCheckingMode;
import eu.emi.security.authn.x509.ValidationError;
import eu.emi.security.authn.x509.ValidationResult;
import eu.emi.security.authn.x509.X509CertChainValidator;
import eu.emi.security.authn.x509.impl.OpensslCertChainValidator;
import eu.emi.security.authn.x509.proxy.ProxyUtils;

/**
 * An authentication stage that validates X.509 Proxy certificate stored in the {@link X509ProxyCertificateContext}.
 */
@Events({
    @Event(id = EventIds.PROCEED_EVENT_ID)})
public class ValidateX509ProxyCertificate extends AbstractAuthenticationAction {
	
    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(ValidateX509ProxyCertificate.class);

    /** {@inheritDoc} */
	protected org.springframework.webflow.execution.Event doExecute(final HttpServletRequest httpRequest, final HttpServletResponse httpResponse,
			final RequestContext springRequestContext, final ProfileRequestContext profileRequestContext,
			final AuthenticationRequestContext authenticationContext) throws AuthenticationException {
		X509Certificate[] certificateChain = authenticationContext.getSubcontext(X509ProxyCertificateContext.class).getCertificateChain();
		if (certificateChain == null || certificateChain.length == 0) {
			throw new AuthenticationException("Could not find the certificate chain from the context!");
		}
		X500Principal principal = ProxyUtils.getOriginalUserDN(certificateChain);
		
		// TODO: make the validator a Spring bean
		X509CertChainValidator vff = new OpensslCertChainValidator(
	        "/etc/grid­security/certificates",
	        NamespaceCheckingMode.EUGRIDPMA_AND_GLOBUS, 60000);
		ValidationResult result = vff.validate(certificateChain);
		if (!result.isValid()) {
	        List<ValidationError> errors = result.getErrors();
	        log.info("Proxy certificate for user {} could not be validated, {} errors found", principal, errors.size());
	        if (log.isDebugEnabled()) {
	        	for (int i = 0; i < errors.size(); i++) {
	        		log.debug("Validation error #{}: {}", i+1, errors.get(i).toString());
	        	}
	        }
	        return ActionSupport.buildEvent(this, ValidateX509Certificate.TRANSITION_INVALID_CERTIFICATE, null);
		}
		log.debug("Setting the authenticated principal as {}", principal);
		authenticationContext.setAuthenticatedPrincipal(principal);
		
		return ActionSupport.buildProceedEvent(this);

	}
}
