/*
 * Copyright (c) Members of the EMI Collaboration. 2010-2012.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0 
 */
package org.glite.sts.authn;

import java.security.cert.X509Certificate;
import java.util.List;

import javax.annotation.Nonnull;
import javax.security.auth.x500.X500Principal;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.shibboleth.ext.spring.webflow.Event;
import net.shibboleth.ext.spring.webflow.Events;
import net.shibboleth.idp.authn.AbstractAuthenticationAction;
import net.shibboleth.idp.authn.AuthenticationException;
import net.shibboleth.idp.authn.AuthenticationRequestContext;
import net.shibboleth.idp.authn.X509CertificateContext;
import net.shibboleth.idp.profile.ActionSupport;
import net.shibboleth.idp.profile.EventIds;
import net.shibboleth.idp.profile.ProfileRequestContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.webflow.execution.RequestContext;

import eu.emi.security.authn.x509.NamespaceCheckingMode;
import eu.emi.security.authn.x509.ValidationError;
import eu.emi.security.authn.x509.ValidationResult;
import eu.emi.security.authn.x509.X509CertChainValidator;
import eu.emi.security.authn.x509.impl.OpensslCertChainValidator;

/**
 * An authentication stage that validates {@link X509Certificate} stored in the {@link X509CertificateContext}.
 */
@Events({
    @Event(id = EventIds.PROCEED_EVENT_ID)})
public class ValidateX509Certificate extends AbstractAuthenticationAction {

    /** Transition name returned when the certificate cannot be validated or is invalid. */
    public static final String TRANSITION_INVALID_CERTIFICATE = "InvalidCertificate";

    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(ValidateX509Certificate.class);

    /** {@inheritDoc} */
    protected org.springframework.webflow.execution.Event doExecute(@Nonnull final HttpServletRequest httpRequest,
            @Nonnull final HttpServletResponse httpResponse,
            @Nonnull final RequestContext springRequestContext, 
            @Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationRequestContext authenticationContext) throws AuthenticationException {

        X509Certificate[] certificate = 
            { authenticationContext.getSubcontext(X509CertificateContext.class).getCertificate() };
        X500Principal principal = certificate[0].getSubjectX500Principal();
        // TODO: make the validator a Spring bean
        X509CertChainValidator vff = new OpensslCertChainValidator("/etc/grid­security/certificates",
                NamespaceCheckingMode.EUGRIDPMA_AND_GLOBUS, 60000);
        ValidationResult result = vff.validate(certificate);
        if (!result.isValid()) {
            List<ValidationError> errors = result.getErrors();
            log.info("Certificate {} could not be validated", principal);
            if (log.isDebugEnabled()) {
                for (int i = 0; i < errors.size(); i++) {
                    log.debug("Validation error #{}: {}", i+1, errors.get(i).toString());
                }
            }
            return ActionSupport.buildEvent(this, TRANSITION_INVALID_CERTIFICATE, null);
        }
        log.debug("Setting the authenticated principal as {}", principal);
        authenticationContext.setAuthenticatedPrincipal(principal);
        return ActionSupport.buildProceedEvent(this);
    }
}
