/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.ui.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.PushbackInputStream;
import java.util.Arrays;

/**
 * PasswordReader reads a password and masks the typed inputs.
 * 
 * See article: Password Masking in the Java Programming Language
 * http://java.sun.com/developer/technicalArticles/Security/pwordmask/
 * 
 * Usage:
 * 
 * <pre>
 * char[] password= PasswordReader.getPassword(System.in, &quot;Password: &quot;);
 * </pre>
 * 
 * Imported from <code>org.glite.slcs.ui.util.PasswordReader</code>
 * 
 * @author Valery Tschopp <tschopp@switch.ch>
 */
public class PasswordReader implements Runnable {
	
	/**
     * Gets the password input from <code>InputStream</code>.
     * @param in the user input
     * @param prompt the prompt for the user
     * @return given password
     * @throws IOException if the input cannot be understood
     */
    public static final char[] getPassword(InputStream in, String prompt)
            throws IOException {
        PasswordReader maskingThread= new PasswordReader(prompt);
        maskingThread.setEchoChar(' ');
        Thread thread= new Thread(maskingThread);
        thread.start();

        char[] lineBuffer;
        char[] buf;

        buf= lineBuffer= new char[128];

        int room= buf.length;
        int offset= 0;
        int c;

        loop: while (true) {
            switch (c= in.read()) {
            case -1:
            case '\n':
                break loop;

            case '\r':
                int c2= in.read();
                if ((c2 != '\n') && (c2 != -1)) {
                    if (!(in instanceof PushbackInputStream)) {
                        in= new PushbackInputStream(in);
                    }
                    ((PushbackInputStream) in).unread(c2);
                }
                else {
                    break loop;
                }

            default:
                if (--room < 0) {
                    buf= new char[offset + 128];
                    room= buf.length - offset - 1;
                    System.arraycopy(lineBuffer, 0, buf, 0, offset);
                    Arrays.fill(lineBuffer, ' ');
                    lineBuffer= buf;
                }
                buf[offset++]= (char) c;
                break;
            }
        }
        maskingThread.stopMasking();
        if (offset == 0) {
            return null;
        }
        char[] ret= new char[offset];
        System.arraycopy(buf, 0, ret, 0, offset);
        Arrays.fill(buf, ' ');
        return ret;
    }

    private volatile boolean masking;

    private char echochar= '*';

    /**
     * @param prompt
     *            The prompt displayed to the user
     */
    private PasswordReader(String prompt) {
        System.out.print(prompt);
        System.out.flush();
        // do masking by default
        masking= true;
    }

    /**
     * Begin masking until asked to stop.
     */
    public void run() {

        int priority= Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(Thread.MAX_PRIORITY);

        try {
            while (masking) {
                System.out.print("\010" + echochar);
                try {
                    // attempt masking at this rate
                    Thread.sleep(1);
                } catch (InterruptedException iex) {
                    Thread.currentThread().interrupt();
                    return;
                }
            }
        } finally { // restore the original priority
            Thread.currentThread().setPriority(priority);
        }
    }

    /**
     * Instruct the thread to stop masking.
     */
    private void stopMasking() {
        this.masking= false;
    }

    /**
     * Sets the char to replace input
     */
    private void setEchoChar(char c) {
        this.echochar= c;
    }

}
