/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.ui.client.message;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.security.GeneralSecurityException;
import java.util.Iterator;

import net.htmlparser.jericho.Segment;
import net.htmlparser.jericho.StreamedSource;

import org.apache.http.HttpResponse;
import org.glite.pseudo.common.pki.Certificate;
import org.glite.pseudo.ui.client.PseudoServiceClientException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is a very simple wrapper to the CertResponse -message, issued by the Pseudonymity Service.
 * It contains the status message and the certificate (chain).
 * The XML-message is NOT throughly validated, only the elements are parsed.
 */
public class CertResponse {
	
	/** Logging */
    private static final Logger log = LoggerFactory.getLogger(CertResponse.class);
    
    /** The status element from the message*/
    protected String status;
    
    /** The certificate element from the message */
    protected String certificate;
    
    /**
     * Constructs a new <code>CertResponse</code>
     */
    protected CertResponse() {
    	// no op
    }
    
	/**
	 * Constructs a new <code>CertResponse</code>.
	 * @param httpResponse the <code>HttpResponse</code> object from the pseudonymity service
	 * @throws PseudoServiceClientException if the response object could not be parsed
	 */
    public CertResponse(HttpResponse httpResponse) throws PseudoServiceClientException {
		log.debug("Initializing CertResponse");
		if (httpResponse == null) {
			throw new PseudoServiceClientException("Error while parsing the response message: the response is empty!");
		}
		try {
			InputStream inputStream = httpResponse.getEntity().getContent();
			parseStream(inputStream);
			inputStream.close();
		} catch (IOException e) {
			throw new PseudoServiceClientException("Error while parsing the response message!", e);
		}
		log.trace("CertResponse successfully initialized: {}", this.toString());
	}

	/**
	 * Parses the CertResponse-elements from <code>InputStream</code>.
	 * @param inputStream the input
	 * @throws IOException if the input could not be parsed
	 */
    protected void parseStream(InputStream inputStream) throws IOException {
		StreamedSource source = new StreamedSource(inputStream);
		Iterator<Segment> iterator = source.iterator();
		if (!source.isXML()) {
			throw new IOException("The response is not a valid XML message!");
		}
		while (iterator.hasNext()) {
			Segment segment = iterator.next();
			String segmentStr = segment.toString().trim();
			log.debug("Found segment '{}'", segmentStr);
			if (segmentStr.equals("<Status>")) {
				segment = iterator.next();
				log.debug("Setting status={}", segment.toString().trim());
				this.status = segment.toString().trim();
			} else if (segmentStr.equals("<Certificate format=\"PEM\">")) {
				segment = iterator.next();
				log.debug("Setting certificate={}", segment.toString().trim());
				this.certificate = segment.toString().trim();
			}
		}
	}
	
	/**
	 * Parses the certificate from the response message.
	 * @return the parsed certificate as <code>Certificate</code>
	 * @throws PseudoServiceClientException if the certificate could not be parsed
	 */
    public Certificate getCertificate() throws PseudoServiceClientException {
		if (this.certificate == null || this.certificate.equals("")) {
			throw new PseudoServiceClientException("The certificate is empty!");
		}
		StringReader reader = new StringReader(this.certificate);
		try {
			return Certificate.readPEM(reader);
		} catch (IOException e) {
			throw new PseudoServiceClientException("Could not read the certificate", e);
		} catch (GeneralSecurityException e) {
			throw new PseudoServiceClientException("Failed to reconstitute the certificate", e);
		}
	}
	
	/**
	 * Gets the status message from the response message.
	 * @return the status message as <code>String</code>
	 */
    public String getStatus() {
		return this.status;
	}
	
	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
    public String toString() {
		return "CertResponse: status=" + this.status + ", certificate=" + certificate;
	}

}
