/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.ui.client.message;

import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import net.htmlparser.jericho.Attribute;
import net.htmlparser.jericho.Attributes;
import net.htmlparser.jericho.Segment;
import net.htmlparser.jericho.StreamedSource;

import org.apache.http.HttpResponse;
import org.glite.pseudo.common.pki.CertificateExtension;
import org.glite.pseudo.common.pki.CertificateExtensionFactory;
import org.glite.pseudo.common.pki.CertificateKeys;
import org.glite.pseudo.common.pki.CertificateRequest;
import org.glite.pseudo.ui.client.PseudoServiceClientException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is a very simple wrapper class to the CertInfoResponse-message, issued by the Pseudonymity Service.
 * The XML-message is NOT throughly validated, only the elements are parsed.
 */
public class CertInfoResponse {
	
	/** Logging */
    private static final Logger log = LoggerFactory.getLogger(CertInfoResponse.class);
	
	/** The status of the response message */
    protected String status;
    
    /** The subject DN from the message */
	protected String subjectDN;
	
	/** Authorization token from the message */
	protected String authorizationToken;
	
	/** URL of the certificate end-point of the Pseudonymity Service */
	protected String serviceEndpoint;
	
	/** The list of required certificate extensions for the CSR */
	protected List<CertificateExtension> certificatePolicy;
	
	/**
	 * Constructs a new <code>CertInfoResponse</code>.
	 */
	protected CertInfoResponse() {
		// no op
	}
	
	/**
	 * Constructs a new <code>CertInfoResponse</code>.
	 * @param httpResponse the <code>HttpResponse</code> object from the pseudonymity service
	 * @throws PseudoServiceClientException if the response object could not be parsed
	 */
	public CertInfoResponse(HttpResponse httpResponse) throws PseudoServiceClientException {
		if (httpResponse == null) {
			throw new PseudoServiceClientException("Error while parsing the response message: the response is empty!");
		}
		try {
			InputStream inputStream = httpResponse.getEntity().getContent();
			parseStream(inputStream);
			inputStream.close();
		} catch (IOException e) {
			throw new PseudoServiceClientException("Error while parsing the response message!", e);
		}
		log.debug("Parsed the following information: {}", this.toString());
	}
	
	/**
	 * Parses the CertInfoResponse-elements from <code>InputStream</code>.
	 * @param inputStream the input
	 * @throws IOException if the input could not be parsed
	 */
	protected void parseStream(InputStream inputStream) throws IOException {
		StreamedSource source = new StreamedSource(inputStream);
		Iterator<Segment> iterator = source.iterator();
		if (!source.isXML()) {
			throw new IOException("The response is not a valid XML message!");
		}
		while (iterator.hasNext()) {
			Segment segment = iterator.next();
			String segmentStr = segment.toString().trim();
			log.debug("Found segment '{}'", segmentStr);
			if (segmentStr.equals("<Status>")) {
				segment = iterator.next();
				log.debug("Setting status={}", segment.toString().trim());
				this.status = segment.toString().trim();
			} else if (segmentStr.equals("<AuthorizationToken>")) {
				segment = iterator.next();
				log.debug("Setting authorizationToken={}", segment.toString().trim());
				this.authorizationToken = segment.toString().trim();
			} else if (segmentStr.equals("<SubjectDN>")) {
				segment = iterator.next();
				log.debug("Setting subjectDN={}", segment.toString().trim());
				this.subjectDN = segment.toString().trim();
			} else if (segmentStr.equals("<ServiceEndpoint>")) {
				segment = iterator.next();
				log.debug("Setting serviceEndpoint={}", segment.toString().trim());
				this.serviceEndpoint = segment.toString().trim();
			} else if (segmentStr.equals("<CertificatePolicy>")) {
				iterator = readCertificatePolicy(iterator);
			}
		}
	}
	
	private Iterator<Segment> readCertificatePolicy(Iterator<Segment> iterator) {
		String segmentStr;
		this.certificatePolicy = new ArrayList<CertificateExtension>();
		do {
			String name = null;
			String oid = null;
			boolean critical = false;
			Segment segment = iterator.next();
			segmentStr = segment.toString().trim();
			if (segmentStr.length() > 21 && segmentStr.substring(0, 22).equals("<CertificateExtension ")) {
				log.debug("Segment={}", segment);
			
				Attributes attrs = segment.parseAttributes();
				for (int i = 0; i < attrs.size(); i++) {
					Attribute attribute = attrs.get(i);
					String key = attribute.getKey();
					String value = attribute.getValue();
					if (key.equals("name")) {
						name = value;
					} else if (key.equals("oid")) {
						oid = value;
					} else if (key.equals("critical")) {
						critical = new Boolean(value).booleanValue();
					} else {
						log.warn("Unrecognized attribute in the CertificateExtension");
					}
			}	
				segment = iterator.next();
				segmentStr = segment.toString().trim();
				CertificateExtension extension = CertificateExtensionFactory.createCertificateExtension(name, segmentStr, critical);
				if (extension.getOID().toString().equals(oid)) {
					log.debug("Adding the extension to the policy");
					this.certificatePolicy.add(extension);
				} else {
					log.warn("The OID did not match, the extension has been ignored!");
				}
			}
		} while (!segmentStr.equals("</CertificatePolicy>"));
		return iterator;
	}

	/**
	 * Generates the certificate request using the given key-pair.
	 * @param certKeys the key-pair used for the CSR
	 * @return certificate signing request
	 * @throws PseudoServiceClientException if the CSR could not be generated
	 */
	public CertificateRequest generateCSR(CertificateKeys certKeys) throws PseudoServiceClientException {
		log.debug("Generating a CertificateRequest object");
		CertificateRequest csr = null;
	    try {
			csr = new CertificateRequest(certKeys, this.subjectDN, this.certificatePolicy);
		} catch (GeneralSecurityException e) {
			throw new PseudoServiceClientException("Could not generate CertificateRequest!", e);
		}
        return csr;
	}
	
	/**
	 * Gets the status of the response message.
	 * @return the status of the message
	 */
	public String getStatus() {
		return this.status;
	}
	
	/**
	 * Gets the authorization token of the response message
	 * @return the authorization token of the message
	 */
	public String getAuthorizationToken() {
		return this.authorizationToken;
	}
	
	/**
	 * Gets the certificate service end-point
	 * @return the service end-point
	 */
	public String getServiceEndpoint() {
		return this.serviceEndpoint;
	}
	
	/**
	 * Gets the required certificate extensions for the CSR
	 * @return the list of required certificate extensions
	 */
	public List<CertificateExtension> getCertificateExtensions() {
		return this.certificatePolicy;
	}
	
	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return "CertInfoResponse: Status=" + status + ", SubjectDN=" + subjectDN + ", AuthorizationToken=" + authorizationToken + ", ServiceEndpoint=" + serviceEndpoint + ", CertificatePolicy=" + certificatePolicy;
	}

}
