/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.ui.client;

import java.io.IOException;
import java.net.URL;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.List;

import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HTTP;
import org.glite.pseudo.common.http.HttpClientBuilder;
import org.glite.pseudo.common.pki.CertificateRequest;
import org.glite.pseudo.ui.client.message.CertInfoResponse;
import org.glite.pseudo.ui.client.message.CertResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class handles the communication with the Pseudonymity Service.
 */
public class PseudoServiceClient {
	
	/** Logging */
    private static final Logger log = LoggerFactory.getLogger(PseudoServiceClient.class);
    
    /** Grid proxy file used for the client authentication */
    protected String gridProxyFile;
    
    /** The directory for the CA files */
    protected String caDirectory;
    
    /** Truststore file */
    protected String truststore;
    
    /** Truststore password */
    protected String truststorePwd;
    
    /** Require CRL check during the server certificate validation? */
    protected boolean crlRequired;
    
    /**
     * Constructs a new <code>PseudoServiceClient</code>.
     * @param gridProxyFile the full path for the user's grid proxy file
     * @param caDirectory the directory for the CA files
     * @param crlRequired should the CRL be required for validating the server certificate
     */
    public PseudoServiceClient(String gridProxyFile, String caDirectory, boolean crlRequired) {
    	this.gridProxyFile = gridProxyFile;
    	this.caDirectory = caDirectory;
    	this.crlRequired = crlRequired;
    }
    
    /**
     * Constructs a new <code>PseudoServiceClient</code>.
     * @param gridProxyFile the full path for the user's grid proxy file
     * @param truststore the full path for the truststore file
     * @param truststorePwd the password for the truststore
     * @param crlRequired should the CRL be required for validating the server certificate
     */
    public PseudoServiceClient(String gridProxyFile, String truststore, String truststorePwd, boolean crlRequired) {
    	this.gridProxyFile = gridProxyFile;
    	this.truststore = truststore;
    	this.truststorePwd = truststorePwd;
    	this.crlRequired = crlRequired;
    }
	
	/**
	 * Logins to the Pseudonymity Service.
	 * @param serviceUrl the service endpoint
	 * @return the response message from the service as <code>CertInfoResponse</code>
	 * @throws PseudoServiceClientException if communication errors occured
	 */
    public CertInfoResponse login(URL serviceUrl) throws PseudoServiceClientException {
		HttpClientBuilder httpBuilder = new HttpClientBuilder(serviceUrl, this.gridProxyFile);
		HttpClient httpClient;
		try {
			httpClient = httpBuilder.buildClient();
		} catch (IOException e) {
			throw new PseudoServiceClientException("Could not read the required files for initializing the Http client!", e);
		} catch (GeneralSecurityException e) {
			throw new PseudoServiceClientException("Security-error while initializing the Http client!", e);
		}
		HttpGet httpRequest = new HttpGet(serviceUrl.toString());
		HttpResponse httpResponse = null;
		CertInfoResponse result = null;
		try {
			httpResponse = httpClient.execute(httpRequest);
			result = new CertInfoResponse(httpResponse);
		} catch (ClientProtocolException e) {
			log.error("Could not login to the Pseudonymity Service", e);
			throw new PseudoServiceClientException("Protocol error while contacting the service!", e);
		} catch (IOException e) {
			log.error("Could not login to the Pseudonymity Service", e);
			throw new PseudoServiceClientException("IO error while contacting the service!", e);
		} finally {
			httpClient.getConnectionManager().shutdown();
		}
		return result;
	}
	
	/**
	 * Sends the certificate request to the Pseudonymity Service in order to obtain the pseudonymous certificate.
	 * @param csr the certificate request
	 * @param serviceUrl the service end-point
	 * @param authorizationToken authorization token obtained during the login phase
	 * @return the response message from the service as <code>CertResponse</code>
	 * @throws PseudoServiceClientException if communication errors occured
	 */
    public CertResponse sendCertRequest(CertificateRequest csr, URL serviceUrl, String authorizationToken) throws PseudoServiceClientException {
		HttpClientBuilder httpBuilder = new HttpClientBuilder(serviceUrl, this.gridProxyFile);
		HttpClient httpClient = null;
		try {
			httpClient = httpBuilder.buildClient();
		} catch (IOException e) {
			throw new PseudoServiceClientException("Could not read the required files for initializing the Http client!", e);
		} catch (GeneralSecurityException e) {
			throw new PseudoServiceClientException("Security-error while initializing the Http client!", e);
		}
		HttpPost httpRequest = new HttpPost(serviceUrl.toString());
		List<NameValuePair> httpRequestParams = new ArrayList<NameValuePair>();
		httpRequestParams.add(new BasicNameValuePair("AuthorizationToken", authorizationToken));
		httpRequestParams.add(new BasicNameValuePair("CertificateSigningRequest", csr.getPEMEncoded()));
		HttpResponse httpResponse = null;
		CertResponse result = null;
		try {
			httpRequest.setEntity(new UrlEncodedFormEntity(httpRequestParams, HTTP.UTF_8));
			httpResponse = httpClient.execute(httpRequest);
			result = new CertResponse(httpResponse);
		} catch (ClientProtocolException e) {
			log.error("Could not obtain the certificate from the Pseudonymity Service", e);
			throw new PseudoServiceClientException("Protocol error while contacting the service!", e);
		} catch (IOException e) {
			log.error("Could not obtain the certificate from the Pseudonymity Service", e);
			throw new PseudoServiceClientException("IO error while contacting the service!", e);
		} finally {
			httpClient.getConnectionManager().shutdown();	
		}
		return result;
	}
    
}
