/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.ui;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.GeneralSecurityException;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.glite.pseudo.common.log.LogbackLoggingService;
import org.glite.pseudo.common.pki.Certificate;
import org.glite.pseudo.common.pki.CertificateKeys;
import org.glite.pseudo.common.util.Util;
import org.glite.pseudo.ui.client.PseudoServiceClient;
import org.glite.pseudo.ui.client.PseudoServiceClientException;
import org.glite.pseudo.ui.client.message.CertInfoResponse;
import org.glite.pseudo.ui.client.message.CertResponse;
import org.glite.pseudo.ui.util.PasswordReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Command line client tool for the Pseudonymity Service.
 * 
 * By default, the service certificate is validated using the CA-files in the /etc/grid-security/certificates -directory.
 * Truststores are also supported via the command line parameters.
 */
public class PseudoInit {

    /** The default keysize for the public/private keypair */
    private static final int DEFAULT_KEYSIZE = 1024;

    /** The minimum keysize for the public/private keypair */
    private static final int MINIMUM_KEYSIZE = 512;

    /** Postfix for the certificate file */
    private static final String PSEUDO_CERTIFICATE_FILE_POSTFIX = "cert.pem";

    /** Postfix for the private key file */
    private static final String PSEUDO_PRIVATE_KEY_FILE_POSTFIX = "key.pem";

    /** Default prefix for the certificate and private key files */
    private static final String DEFAULT_FILE_PREFIX = "pseudo";

    /** Default subdirectory for the credentials under the user home directory */
    private static final String DEFAULT_STORE_DIRECTORY = ".globus";

    /** Logging */
    private static final Logger log = LoggerFactory.getLogger(PseudoInit.class);

    /** Private and public keys */
    private CertificateKeys m_certificateKeys = null;

    /** The URL for the pseudonymity service's login servlet */
    private URL m_serviceUrl = null;

    /** The keysize for the public/private keypair */
    private int m_keySize = -1;

    /** The encryption password for the private key */
    private char[] m_keyPassword = null;

    /** The absolute path to the user proxy */
    private String m_proxyFile = null;

    /** The absolute path to the (pseudonymous) private key file */
    private File m_keyfile = null;

    /** The absolute path to the (pseudonymous) certificate file */
    private File m_certfile = null;

    /** If true, more detailed output is provided */
    private boolean m_verbose = false;

    /** The truststore used for authenticating the pseudonymity service */
    private String m_truststore = null;
    
    /** The truststore password */
    private String m_truststorePwd = null;
    
    /** The directory for the files related to the CAs */
    private String m_caDirectory = "/etc/grid-security/certificates";
    
    /** Check CRL during the server certificate validation */
    private boolean m_checkCrl = true;

    /**
     * <p>
     * Constructs a new <code>PseudoInit</code>.
     * </p>
     */
    protected PseudoInit() {
        // no op
    }

    /**
     * <p>
     * The controller method for the program sequence.
     * </p>
     * 
     * @param args
     *            the commandline arguments.
     */
    public static void main(String[] args) {
    	
    	// Initialize the logging service
    	new LogbackLoggingService("/opt/glite/pseudoclient/logging.xml");

        // display program + copyright information
        System.out.println("pseudo-cert-request: " + PseudoInit.class.getName() + " - "
                + PseudoClientVersion.COPYRIGHT);
        System.out.println();

        // initialize the command line
        String cmdLine = "";
        for (int i = 0; i < args.length; i++) {
            cmdLine = cmdLine + args[i] + " ";
        }
        log.info("Program started with the following arguments: " + cmdLine);

        // construct a new client
        PseudoInit client = new PseudoInit();

        // read some user properties from the system
        String userHome = System.getProperty("user.home");
        String separator = System.getProperty("file.separator");
        // the following variable should be set by the startup script
        String userUid = System.getProperty("userid");
        String defaultDir = userHome + separator + DEFAULT_STORE_DIRECTORY;
        // /tmp/x509up<uid>
        if (userUid != null) {
        	client.m_proxyFile = separator + "tmp" + separator + "x509up_u" + userUid;
        } else {
        	client.m_proxyFile = "x509up_u";
        }

        // parse command line and configure the client
        Options options = createCommandLineOptions(defaultDir, client.m_proxyFile);
        client.parseCommandLineOptions(options, args, defaultDir);

        // connect the pseudonymity service to obtain CertInfoResponse XML
        if (client.m_verbose) {
            System.out.print("Connecting to the pseudonymity service ...");
        }
        CertInfoResponse certInfo = null;
        PseudoServiceClient serviceClient = null;
        if (client.m_truststore == null) {
        	serviceClient = new PseudoServiceClient(client.m_proxyFile, client.m_caDirectory, client.m_checkCrl);
        } else {
        	serviceClient = new PseudoServiceClient(client.m_proxyFile, client.m_truststore, client.m_truststorePwd, client.m_checkCrl);
        }
        try {
        	certInfo = serviceClient.login(client.m_serviceUrl);
        } catch (Exception e) {
            errorExit("Error while connecting pseudonymity service!", e, client.m_verbose);
        }

        if (client.m_verbose) {
            System.out.println(" ok");
            System.out.print("Generating a public/private keypair ("
                    + client.m_keySize + " bit) ...");
        }

        // generate the keypair and the certificate request
        try {
            client.m_certificateKeys = new CertificateKeys(client.m_keySize,
                    client.m_keyPassword);
        } catch (GeneralSecurityException e) {
            errorExit("unable to generate a certificate request!", e, client.m_verbose);
        }

        if (client.m_verbose) {
            System.out.println(" ok");
            System.out.print("Submitting the certificate request ...");
        }

        // submit the certificate request to the pseudonymity service
        CertResponse certResponse = null;
        try {
        	URL endpoint = new URL(certInfo.getServiceEndpoint());
        	certResponse = serviceClient.sendCertRequest(certInfo.generateCSR(client.m_certificateKeys), endpoint, certInfo.getAuthorizationToken());
        } catch (Exception e) {
            errorExit("could not request a certificate from the service!", e, client.m_verbose);
        }

        if (client.m_verbose) {
            System.out.println(" ok");
        }

        // Save the certificate to the filesystem
        Certificate certificate = null;
        try {
        	certificate = certResponse.getCertificate();
            certificate.storePEM(client.m_certfile);
        } catch (PseudoServiceClientException e) {
			errorExit("Could not read the certificate from the response!", e, client.m_verbose);
        } catch (IOException e) {
            errorExit("Unable to save the certificate!", e, client.m_verbose);
        }

        // Save the private key to the filesystem
        try {
            client.m_certificateKeys.storePEMPrivate(client.m_keyfile);
        } catch (IOException e) {
            errorExit("Unable to save the private key!", e, client.m_verbose);
        }

        // Certificate & key saved to the filesystem, printing the final info
        // to the user
        System.out.println();
        System.out.println("Pseudo certificate ("
                + client.m_certfile.getAbsolutePath() + ") expires on '"
                + certificate.getCertificate().getNotAfter() + "'.");

        // All done. Exiting program with normal status (0)
        System.exit(0);
    }

    /**
     * <p>
     * Parses the command line options and sets the corresponding class
     * variables.
     * </p>
     * 
     * @param options
     *            the command line options
     * @param args
     *            the command line arguments
     * @param defaultDir
     *            the default store directory for the certificate and private
     *            key files
     */
    private void parseCommandLineOptions(Options options, String[] args, String defaultDir) {
        CommandLineParser parser = new PosixParser();
        CommandLine cmd = null;
        try {
            cmd = parser.parse(options, args);
        } catch (ParseException e) {
            System.err.println("ERROR: " + e.getMessage());
            printHelp(options);
            System.exit(1);
        }

        // version?
        if (cmd.hasOption('V')) {
            System.out.println("Version: " + PseudoClientVersion.getVersion());
            System.exit(0);
        }

        // help? or error
        if (cmd.hasOption('h')) {
            printHelp(options);
            System.exit(1);
        }

        // verbose?
        if (cmd.hasOption('v')) {
            this.m_verbose = true;
        }
        
        if (cmd.hasOption('a')) {
        	this.m_truststore = cmd.getOptionValue('a');
        	if (!cmd.hasOption('b')) {
        		System.err.println("ERROR: --truststorepwd must also be defined with --truststore option!");
        		printHelp(options);
        		System.exit(1);
        	} else {
        		this.m_truststorePwd = cmd.getOptionValue('b');
        	}
        }
        
        if (cmd.hasOption('n')) {
        	this.m_checkCrl = false;
        }
        
        if (cmd.hasOption('c')) {
        	this.m_caDirectory = cmd.getOptionValue('c');
        }

        // pseudonymity service url
        String serviceUrlStr = null;
        if (cmd.hasOption('u')) {
            serviceUrlStr = cmd.getOptionValue('u');
            if (serviceUrlStr == null) {
                errorExit("--url: empty pseudonymity service url");
            } else {
                try {
                    this.m_serviceUrl = new URL(serviceUrlStr);
                } catch (MalformedURLException e) {
                    errorExit("--url: " + e);
                }
                if (!this.m_serviceUrl.getProtocol().equals("https")) {
                    errorExit("pseudonymity service protocol ("
                            + this.m_serviceUrl.getProtocol()
                            + ") must be https!");
                }
            }
        } else {
            errorExit(" mandatory option --url <service-url> is missing");
        }
        log.debug("ServiceUrl: " + this.m_serviceUrl);
        if (this.m_verbose) {
            System.out.println("ServiceUrl: " + this.m_serviceUrl);
        }

        // user prefix
        String userPrefix = null;
        if (cmd.hasOption('P')) {
            userPrefix = cmd.getOptionValue('P');
            if (userPrefix == null) {
                errorExit("--prefix: empty prefix");
            }
            if (this.m_verbose) {
                System.out.println("Prefix: " + userPrefix);
            }
        } else {
            userPrefix = DEFAULT_FILE_PREFIX;
        }

        // store directory
        String storeDirectory = null;
        if (cmd.hasOption('D')) {
            storeDirectory = cmd.getOptionValue('D');
            if (storeDirectory == null) {
                errorExit("--storedir: empty store dirname");
            }
            if (this.m_verbose) {
                System.out.println("Store Directory: " + storeDirectory);
            }
        } else {
            storeDirectory = defaultDir;
        }

        this.checkStoreDirectory(storeDirectory, userPrefix);

        // private key size

        if (cmd.hasOption('s')) {
            this.m_keySize = Integer.parseInt(cmd.getOptionValue('s'));
        } else {
            this.m_keySize = DEFAULT_KEYSIZE;
        }
        if (this.m_keySize < MINIMUM_KEYSIZE) {
            errorExit("--keysize <size>, invalid keysize (minimum "
                    + MINIMUM_KEYSIZE + ")");
        }

        // private key password

        if (cmd.hasOption('k')) {
            this.m_keyPassword = cmd.getOptionValue('k').toCharArray();
            if (this.m_verbose) {
                System.out
                        .println("Private key password given in the arguments");
            }
        } else {
            // read the password from console
            boolean match = false;
            while (!match) {
                try {
                    this.m_keyPassword = PasswordReader.getPassword(System.in,
                            "New Key Password: ");
                    char[] retype = PasswordReader.getPassword(System.in,
                            "Retype: ");
                    if (Util.matchCharArrays(this.m_keyPassword, retype)) {
                        match = true;
                    } else {
                        System.out.println("Passwords didn't match!");
                    }
                } catch (IOException e) {
                    // ignored?
                    log.error("IOException caught!", e);
                }
            }
        }
        if (this.m_keyPassword == null) {
            printHelp(options);
            System.exit(1);
        }

        // proxy filename
        if (cmd.hasOption('p')) {
            this.m_proxyFile = cmd.getOptionValue('p');
            if (this.m_proxyFile == null) {
                errorExit("--proxyfile: empty proxy filename");
            }
        } 
        if (this.m_verbose) {
            System.out.println("Proxy Filename: " + this.m_proxyFile);
        }
    }

    /**
     * <p>
     * Prepares the store directory for the certificate and the private key
     * files and sets the corresponding class variables.
     * </p>
     * 
     * @param storeDirectory
     *            the store directory for the certificate and the private key
     *            files
     * @param prefix
     *            the prefix for the certificate and the private key files
     */
    private void checkStoreDirectory(String storeDirectory, String prefix) {
        File directory = new File(storeDirectory);
        if (!directory.isDirectory()) {
            errorExit("directory ( " + storeDirectory + ") does not exist!");
        }
        prefix = storeDirectory + System.getProperty("file.separator") + prefix;
        String certFilename = prefix + PSEUDO_CERTIFICATE_FILE_POSTFIX;
        String keyFilename = prefix + PSEUDO_PRIVATE_KEY_FILE_POSTFIX;
        this.m_certfile = new File(certFilename);
        try {
            // if the file exists and its not 0-byte
            if (!this.m_certfile.createNewFile()
                    && this.m_certfile.length() > 0) {
                errorExit("certificate file (" + certFilename
                        + ") already exists!");
            }
        } catch (IOException e) {
            errorExit("unable to create certfile (" + certFilename
                    + "), check permissions.", e, false);
        }
        this.m_keyfile = new File(keyFilename);
        try {
            // if the file exists and its not 0-byte
            if (!this.m_keyfile.createNewFile() && this.m_keyfile.length() > 0) {
                errorExit("private key file (" + keyFilename
                        + ") already exists!");
            }
        } catch (IOException e) {
            errorExit("unable to create keyfile (" + keyFilename
                    + "), check permissions.", e, false);
        }
    }
    
    /**
     * <p>
     * Creates the command line options.
     * </p>
     * 
     * @param defaultDir
     *            the default directory for storing the certificate and the
     *            private key files
     * @return the command line options
     */
    private static Options createCommandLineOptions(String defaultDir, String defaultProxy) {
        Option help = new Option("h", "help", false, "this help");
        Option verbose = new Option("v", "verbose", false, "verbose");
        Option version = new Option("V", "version", false, "shows the version");
        Option keysize = new Option("s", "keysize", true,
                "private key size (default: " + DEFAULT_KEYSIZE + ")");
        keysize.setArgName("size");
        Option keypassword = new Option("k", "keypass", true,
                "private key password");
        keypassword.setArgName("password");
        Option prefix = new Option("P", "prefix", true,
                "optional pseudocert.pem and pseudokey.pem filename prefix");
        prefix.setArgName("prefix");
        Option storedir = new Option("D", "storedir", true,
                "absolute pathname to the store directory (default: "
                        + defaultDir + ")");
        storedir.setArgName("directory");
        Option proxyFile = new Option("p", "proxyfile", true,
                "absolute pathname to the proxy file (default: "
                        + defaultProxy + ")");
        proxyFile.setArgName("filename");
        Option host = new Option("u", "url", true, "pseudonymity service url");
        host.setArgName("service-url");
        
        Option truststoreFile = new Option("a", "truststore", true,
                "absolute pathname to the truststore");
        truststoreFile.setArgName("filename");
        Option truststorePwd = new Option("b", "truststorePwd", true, "truststore password");
        truststorePwd.setArgName("password");
        
        Option caDirectory = new Option("c", "cadir", true, "directory for the CA files (default: /etc/grid-security/certificates)");
        caDirectory.setArgName("directory");
        
        Option requireCrl = new Option("n", "nocrl", false, "ignore CRL check of the service certificate");

        Options options = new Options();
        options.addOption(help);
        options.addOption(verbose);
        options.addOption(version);
        options.addOption(keysize);
        options.addOption(keypassword);
        options.addOption(prefix);
        options.addOption(storedir);
        options.addOption(proxyFile);
        options.addOption(host);
        options.addOption(truststoreFile);
        options.addOption(caDirectory);
        options.addOption(requireCrl);
        options.addOption(truststorePwd);
        return options;
    }

    /**
     * <p>
     * Prints help screen with the given options.
     * </p>
     * 
     * @param options
     *            the options included in the help screen.
     */
    private static void printHelp(Options options) {
        HelpFormatter help = new HelpFormatter();
        help.printHelp("pseudo-cert-request --url <service-url> [options]", options);
    }

    /**
     * <p>
     * Exits the program with a specified message.
     * </p>
     * 
     * @param message
     *            the message to be shown to the user
     */
    private static void errorExit(String message) {
        errorExit(message, null, false);
    }

    /**
     * <p>
     * Exits the program with a specified message and exception information.
     * </p>
     * 
     * @param message
     *            the message to be shown to the user
     * @param e
     *            the exception to be dumped to log file
     */
    private static void errorExit(String message, Exception e, boolean verbose) {
        System.err.println("ERROR: " + message);
        if (e == null) {
            log.error(message);
        } else {
            log.error(message, e);
            if (verbose) {
            	e.printStackTrace();
            }
        }
        System.exit(1);
    }
}
