package org.glite.pseudo.server.util;

import java.util.Collection;

import javax.servlet.ServletContext;
import javax.xml.rpc.ServiceException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.web.context.WebApplicationContext;

/**
 * A simple service that exports Spring beans into the Servlet context as an
 * attribute.
 * 
 * Imported from
 * <code>edu.internet2.middleware.shibboleth.common.config.service.ServletContextAttributeExporter</code>
 * .
 * */
public class ServletContextAttributeExporter implements
        ApplicationContextAware, BeanNameAware {

    /** Class logger. */
    private final Logger log = LoggerFactory
            .getLogger(ServletContextAttributeExporter.class);

    /** Application context into which we're loaded. */
    private ApplicationContext appCtx;

    /** ID of this service. */
    private String id;

    /** Whether this service has been initialized. */
    private boolean initialized;

    /** ID of beans exported into the servlet context. */
    private Collection<String> exportedBeans;

    /**
     * Constructor.
     * 
     * @param beans
     *            ID of beans exported into the servlet context
     */
    public ServletContextAttributeExporter(Collection<String> beans) {
        exportedBeans = beans;
    }

    /** {@inheritDoc} */
    public void destroy() throws ServiceException {

    }

    /** {@inheritDoc} */
    public String getId() {
        return id;
    }

    /** {@inheritDoc} */
    public void initialize() throws ServiceException {
        if (!(appCtx instanceof WebApplicationContext)) {
            log.warn("This service may only be used when services are loaded within a web application context.");
            return;
        }

        Object bean;
        if (exportedBeans != null) {
            WebApplicationContext webAppCtx = (WebApplicationContext) appCtx;
            ServletContext servletCtx = webAppCtx.getServletContext();
            for (String beanId : exportedBeans) {
                bean = webAppCtx.getBean(beanId);
                if (bean != null) {
                    log.debug("Exporting bean {} to servlet context.", beanId);
                    servletCtx.setAttribute(beanId, bean);
                } else {
                    log.warn(
                            "No {} bean located, unable to export it to the servlet context",
                            beanId);
                }
            }
        }

        initialized = true;
    }

    /** {@inheritDoc} */
    public boolean isInitialized() {
        return initialized;
    }

    /** {@inheritDoc} */
    public void setApplicationContext(ApplicationContext context) {
        appCtx = context;
    }

    /** {@inheritDoc} */
    public void setBeanName(String name) {
        id = name;
    }

    /** {@inheritDoc} */
    public boolean isDestroyed() {
        return false;
    }
}
