/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.struts.action;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.struts.action.Action;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.action.ActionRedirect;
import org.glite.pseudo.server.Constants;
import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.auditor.AuditAdmin;
import org.glite.pseudo.server.auditor.Auditor;
import org.glite.pseudo.server.auditor.AuditorException;
import org.glite.pseudo.server.auditor.event.AuditEvent;
import org.glite.pseudo.server.auditor.event.PseudoAuditEvent;
import org.glite.pseudo.server.servlet.HttpServletHelper;
import org.glite.pseudo.server.session.AdminSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * Base-class for most of the actions: redirects the user to the login page if
 * he has not been authenticated.
 * </p>
 */
public abstract class AbstractAction extends Action {

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(AbstractAction.class);

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.struts.action.Action#execute(org.apache.struts.action.
     * ActionMapping, org.apache.struts.action.ActionForm,
     * javax.servlet.http.HttpServletRequest,
     * javax.servlet.http.HttpServletResponse)
     */
    public ActionForward execute(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response)
            throws Exception {

        ActionForward forward = null;
        if (beforeAction(form, request, response)) {
            log.debug("User is authenticated.");
            forward = executeAction(mapping, form, request, response);
            afterAction();
        } else {
            log.debug("User is not authenticated, redirecting back to the home page");
            ActionForward homeForward = mapping.findForward("go.home");
            forward = new ActionRedirect(homeForward);
        }
        return forward;
    }

    /**
     * <p>
     * The logic run before the actual action execution.
     * </p>
     * 
     * @param mapping
     *            The ActionMapping used to select this instance
     * @param form
     *            The optional ActionForm bean for this request (if any)
     * @param request
     *            The HTTP request we are processing
     * @param response
     *            The HTTP response we are creating
     * @return true if the (admin) session exists, false otherwise.
     */
    protected boolean beforeAction(ActionForm form, HttpServletRequest request,
            HttpServletResponse response) {
        AdminSession admin = (AdminSession) request.getSession().getAttribute(
                Constants.ADMIN_USER_SESSION_ID);
        if (admin == null) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * <p>
     * The actual execution method for the Action.
     * </p>
     * 
     * @param mapping
     *            The ActionMapping used to select this instance
     * @param form
     *            The optional ActionForm bean for this request (if any)
     * @param request
     *            The HTTP request we are processing
     * @param response
     *            The HTTP response we are creating
     * @return The forward to which control should be transferred, or null if
     *         the response has been completed.
     * @throws Exception
     *             if the application business logic throws an exception
     * @see org.apache.struts.action.Action#execute(org.apache.struts.action.ActionMapping,
     *      org.apache.struts.action.ActionForm,
     *      javax.servlet.http.HttpServletRequest,
     *      javax.servlet.http.HttpServletResponse)
     */
    protected abstract ActionForward executeAction(ActionMapping mapping,
            ActionForm form, HttpServletRequest request,
            HttpServletResponse response) throws Exception;

    /**
     * <p>
     * The logic run after the actual action execution.
     * </p>
     */
    protected void afterAction() {
    }

    /**
     * <p>
     * Logs the (admin) event to the AuditEventDatabase.
     * </p>
     * 
     * @param level
     *            The event level
     * @param message
     *            The event message
     * @param request
     *            The HTTP request we are processing
     * @throws PseudoServerException
     *             if the auditor throws an exception
     */
    protected static void logEvent(int level, String message,
            HttpServletRequest request) throws PseudoServerException {
        AdminSession admin = (AdminSession) request.getSession().getAttribute(
                Constants.ADMIN_USER_SESSION_ID);
        Auditor auditor = HttpServletHelper.getAuditor(request.getSession()
                .getServletContext());
        AuditEvent event = new PseudoAuditEvent(level, message,
                admin.getAttributes());
        try {
            auditor.logEvent(event);
        } catch (AuditorException e) {
            throw new PseudoServerException("Could not audit the event!", e);
        }
    }

    /**
     * <p>
     * Gets the AuditAdmin implementation.
     * </p>
     * 
     * @return the AuditAdmin implementation
     * @throws PseudoServerException
     *             if any errors occur
     */
    protected static AuditAdmin getAuditDatabaseReader(
            HttpServletRequest request) throws PseudoServerException {
        return HttpServletHelper.getAuditAdmin(request.getSession()
                .getServletContext());
    }

}
