/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.standalone;

import java.io.IOException;
import java.security.KeyStoreException;
import java.security.Security;
import java.security.cert.CertificateException;
import java.util.Collections;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.glite.pseudo.server.PseudoServerException;
import org.mortbay.jetty.Connector;
import org.mortbay.jetty.Handler;
import org.mortbay.jetty.Server;
import org.mortbay.jetty.handler.DefaultHandler;
import org.mortbay.jetty.handler.HandlerCollection;
import org.mortbay.jetty.webapp.WebAppContext;
import org.mortbay.thread.concurrent.ThreadPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import eu.emi.security.authn.x509.X509Credential;
import eu.emi.security.authn.x509.helpers.ssl.CredentialX509KeyManager;
import eu.emi.security.authn.x509.helpers.ssl.SSLTrustManager;
import eu.emi.security.authn.x509.impl.OpensslCertChainValidator;
import eu.emi.security.authn.x509.impl.PEMCredential;

public class PseudonymityService {

    /**
     * 
     * Useful defaults for the standalone service
     * 
     */
    final class PseudoStandaloneServiceDefaults {

        /** The service hostname */
        static final String HOSTNAME = "localhost";

        /** The service port on which the pseudonymity service will listen to requests */
        static final int PORT = 443;

        /** The shutdown service port */
        static final int SHUTDOWN_PORT = 8005;

        /** Max request queue size. -1 means unbounded queue is used. */
        static final int MAX_REQUEST_QUEUE_SIZE = -1;

        /** Max concurrent connections. */
        static final int MAX_CONNECTIONS = 64;

        /** Default certificate path for the service */
        static final String CERTIFICATE_PATH = "/etc/grid-security/hostcert.pem";

        /** Default private key path for the service */
        static final String PRIVATE_KEY_PATH = "/etc/grid-security/hostkey.pem";

        /** Default path for CA certificates */
        static final String CA_PATH = "/etc/grid-security/certificates";

        /** Should CRLs be checked by trustmanager? */
        static final boolean CRL_ENABLED = true;

        /**
         * How frequently the Pseudonymity Service should update CRLs, CAs and namespaces from the filesystem.
         *  
         * The default is 30 minutes, ie. 1000*60*30 milliseconds.
         */
        static final long CRL_UPDATE_INTERVAL = 1000 * 60 * 30;

        /**
         * The directory containing all the CA certificates, CRLs and namespace definitions.
         */
        static final String TRUST_STORE_DIR = "/etc/grid-security/certificates";
        
        /**
         * The full path for the Trustmanager logging configuration
         */
        static final String LOG4J_CONFIGURATION_FILE = "/var/lib/trustmanager-tomcat/log4j-trustmanager.properties";
    }

    //private static final String DEFAULT_WAR_LOCATION = System.getProperty("PSEUDO_HOME") + "/wars/pseudo.war";

    /** Logging */
    private static final Logger log = LoggerFactory.getLogger(PseudonymityService.class);

    /**
     * The option name used by callers to specify where the pseudonymity service should
     * look for the configuration
     */
    private static final String CONF_DIR_OPTION_NAME = "conf-dir";

    /** The configuration directory */
    protected String configurationDir;

    /** The jetty http server */
    protected Server jettyServer;

    /** The jetty webapp context in which the wep application is configured */
    private WebAppContext webappContext;

    /**
     * Constructor. Parses the configuration and starts the server.
     * @param args the command line arguments as passed by the {@link #main(String[])} method
     */
    public PseudonymityService(String[] args) {
        try {
            parseOptions(args);
            Security.addProvider(new BouncyCastleProvider());
            StandaloneConfiguration.initialize(configurationDir);
            configureServer();
            jettyServer.start();
            if (webappContext.getUnavailableException() != null)
                throw webappContext.getUnavailableException();
            jettyServer.join();
        } catch (Throwable e) {
            log.error("The service encountered an error that could not be dealt with, shutting down!");
            log.error(e.getMessage());

            // Also print error message to standard error
            System.err.println("The service encountered an error that could not be dealt with, shutting down!");
            System.err.println("Error: " + e.getMessage());
            e.printStackTrace(System.err);

            if (log.isDebugEnabled())
                log.error(e.getMessage(), e);
            try {
                jettyServer.stop();
            } catch (Exception e1) {
                // Ignoring this exception, exiting in any case
            }
            System.exit(1);
        }

    }


    private Connector configureConnector(String host, int port) throws PseudoServerException {
		String privateKey = getStringFromStandaloneConfiguration("private_key", PseudoStandaloneServiceDefaults.PRIVATE_KEY_PATH);
		String privateKeyPwd = getStringFromStandaloneConfiguration("private_key_password", null);
		String certificate = getStringFromStandaloneConfiguration("certificate", PseudoStandaloneServiceDefaults.CERTIFICATE_PATH);
		X509Credential credential;
		try {	
			if (privateKeyPwd == null) {
				log.debug("Initializing the PEM credential without a password");
				credential = new PEMCredential(privateKey, certificate, null);
			} else {
				log.debug("Initializing the PEM credential with a password");
				credential = new PEMCredential(privateKey, certificate, privateKeyPwd.toCharArray());
			}
		} catch (KeyStoreException e) {
			throw new PseudoServerException("Could not initialize the private key!", e);
		} catch (CertificateException e) {
			throw new PseudoServerException("Could not initialize the certificate!", e);
		} catch (IOException e) {
			throw new PseudoServerException("Could not read the private key and/or certificate from the file", e);
		}
		String trustStoreDir = getStringFromStandaloneConfiguration("trust_store_dir", String.valueOf(PseudoStandaloneServiceDefaults.TRUST_STORE_DIR));		
		try {
			OpensslCertChainValidator validator = new OpensslCertChainValidator(trustStoreDir);
			validator.setUpdateInterval(getLongFromStandaloneConfiguration("crl_update_interval", PseudoStandaloneServiceDefaults.CRL_UPDATE_INTERVAL));
			JettySslSelectChannelConnector connector = new JettySslSelectChannelConnector(new CredentialX509KeyManager(credential), new SSLTrustManager(validator));
			connector.setWantClientAuth(true);
			connector.setNeedClientAuth(true);
			return connector;
		} catch (Exception e) {
			log.error("Error initializing connector: "+e.getMessage());
			if (log.isDebugEnabled())
				log.error("Error initializing connector: "+e.getMessage(),e);
			throw new PseudoServerException(e);
		}
    }


    /**
     * Performs the jetty server configuration
     */
    private void configureServer() throws PseudoServerException {

        log.info("Configuring jetty server...");
        jettyServer = new Server();
        int maxRequestQueueSize = getIntFromStandaloneConfiguration(
                "max_request_queue_size",
                PseudoStandaloneServiceDefaults.MAX_REQUEST_QUEUE_SIZE);
        log.debug("maxRequestQueueSize = {}", maxRequestQueueSize);

        int maxConnections = getIntFromStandaloneConfiguration(
                "max_connections", PseudoStandaloneServiceDefaults.MAX_CONNECTIONS);
        if (maxConnections <= 0) {
            log
            .error("Please specify a positive value for the 'maxConnections' configuration parameter!");
            log.error("Will use the hardcoded default '{}' instead...",
                    PseudoStandaloneServiceDefaults.MAX_CONNECTIONS);
            maxConnections = PseudoStandaloneServiceDefaults.MAX_CONNECTIONS;
        }

        log.info("maxConnections = {}", maxConnections);

        jettyServer.setSendServerVersion(false);
        jettyServer.setSendDateHeader(false);

        BlockingQueue<Runnable> requestQueue;

        if (maxRequestQueueSize < 1) {
            requestQueue = new LinkedBlockingQueue<Runnable>();
        } else {
            requestQueue = new ArrayBlockingQueue<Runnable>(maxRequestQueueSize);
        }

        ThreadPool threadPool = new ThreadPool(5, maxConnections, 60,
                TimeUnit.SECONDS, requestQueue);

        jettyServer.setThreadPool(threadPool);

        // Connectors configuration
        int port = getIntFromStandaloneConfiguration("port",
                PseudoStandaloneServiceDefaults.PORT);

        String host = getStringFromStandaloneConfiguration("hostname",
                PseudoStandaloneServiceDefaults.HOSTNAME);

        jettyServer.setConnectors(new Connector[] { configureConnector(host,port) });    

        JettyShutdownCommand shutdownCommand = new JettyShutdownCommand(jettyServer);

        PseudonymityShutdownAndStatusService.startShutdownAndStatusService(getIntFromStandaloneConfiguration("shutdown_port", 8151), Collections
                .singletonList((Runnable) shutdownCommand));

        webappContext = new WebAppContext();

        webappContext.setContextPath("/" + getStringFromStandaloneConfiguration("webapp_context", "pseudo"));
        webappContext.setWar(getStringFromStandaloneConfiguration("warfile", "pseudo.war"));
        webappContext.setParentLoaderPriority(true);

        HandlerCollection handlers = new HandlerCollection();
        handlers.setHandlers(new Handler[] { webappContext, new DefaultHandler() });

        jettyServer.setHandler(handlers);

    }

    /**
     * Utility method to fetch an int configuration parameter out of the
     * standalone-service configuration
     * @param key the configuration parameter key
     * @param defaultValue a default value in case the parameter is not defined
     * @return the configuration parameter value
     */
    private int getIntFromStandaloneConfiguration(String key, int defaultValue) {
        StandaloneConfiguration conf = null;
        try {
            conf = StandaloneConfiguration.instance();
        } catch (PseudoServerException e) {
            log.error("Configuration is not initialized!", e);
        }
        log.debug("Returning {}", conf.getInt(key, defaultValue));
        return conf.getInt(key, defaultValue);
    }

    /**
     * Utility method to fetch a long configuration parameter out of the
     * standalone-service configuration
     * @param key the configuration parameter key
     * @param defaultValue a default value in case the parameter is not defined
     * @return the configuration parameter value
     */
    private long getLongFromStandaloneConfiguration(String key, long defaultValue) {
        StandaloneConfiguration conf = null;
        try {
            conf = StandaloneConfiguration.instance();
        } catch (PseudoServerException e) {
            log.error("Configuration is not initialized!", e);
        }
        log.debug("Returning {}", conf.getLong(key, defaultValue));
        return conf.getLong(key, defaultValue);
    }


    /**
     * Utility method to fetch a string configuration parameter out of the
     * security configuration
     * @param key the configuration parameter key
     * @param defaultValue a default value in case the parameter is not defined
     * @return the configuration parameter value
     */
    private String getStringFromStandaloneConfiguration(String key,
            String defaultValue) {
        StandaloneConfiguration conf = null;
        try {
            conf = StandaloneConfiguration.instance();
        } catch (PseudoServerException e) {
            log.error("Configuration is not initialized!", e);
        }
        log.debug("Returning {}", conf.getString(key, defaultValue));
        return conf.getString(key, defaultValue);
    }

    /**
     * Parses command line options
     * @param args the command line options
     */
    protected void parseOptions(String[] args) {
        if (args.length > 0) {
            int currentArg = 0;
            while (currentArg < args.length) {
                if (!args[currentArg].startsWith("--")) {
                    usage();
                } else if (args[currentArg].equalsIgnoreCase("--"
                        + CONF_DIR_OPTION_NAME)) {
                    configurationDir = args[currentArg + 1];
                    log.info("Starting Pseudonymity Service with configuration dir: {}",
                            configurationDir);
                    currentArg = currentArg + 2;
                } else
                    usage();
            }
        }
    }

    /**
     * Prints a usage message and exits with status 1
     */
    private void usage() {
        String usage = "PseudonymityService [--" + CONF_DIR_OPTION_NAME + " <confDir>]";
        System.out.println(usage);
        System.exit(1);
    }

    /**
     * Runs the service
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        new PseudonymityService(args);
    }

}
