/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.session.impl;

import java.security.SecureRandom;

import org.apache.commons.ssl.util.Hex;
import org.glite.pseudo.server.session.Session;
import org.glite.pseudo.server.session.SessionManager;
import org.glite.pseudo.server.session.StorageService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class provides a simple implementation for the the
 * <code>SessionManager</code> interface.
 */
public class SessionManagerImpl implements SessionManager {

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(SessionManagerImpl.class);

    /** The <code>StorageService</code> used for storing the sessions */
    private StorageService sessionStore;

    /** The lifetime of the sessions in milliseconds */
    private long sessionLifetime;

    /** Number of random bits within a session token. */
    private final int sessionTokenSize = 32;

    /** A {@link SecureRandom} PRNG to generate session tokens. */
    private final SecureRandom prng = new SecureRandom();

    /**
     * Constructs a new <code>SessionManagerImpl</code>.
     * 
     * @param storageService
     *            the <code>StorageService</code> used for storing the sessions
     * @param lifetime
     *            the lifetime of the sessions in milliseconds
     */
    public SessionManagerImpl(StorageService storageService, long lifetime) {
        log.debug("Initializing SessionManagerImpl..");
        this.sessionStore = storageService;
        this.sessionLifetime = lifetime;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.session.SessionManager#createSession(java.lang
     * .String)
     */
    public Session createSession(String dn) {
        log.debug("Creating a session for DN={}", dn);
        byte[] sid = new byte[sessionTokenSize];
        prng.nextBytes(sid);
        String token = Hex.encode(sid);
        Session session = new SessionImpl(dn, token, this.sessionLifetime);
        log.debug("Putting the session to the storage service");
        this.sessionStore.put(session);
        return session;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.session.SessionManager#getSession(java.lang.String
     * , java.lang.String)
     */
    public Session getSession(String dn, String token) {
        if (token == null) {
            log.error("Token cannot be null! DN was {}", dn);
            return null;
        }
        Session session = this.sessionStore.get(token);
        if (session == null || dn == null) {
            return null;
        } else {
            if (dn.trim().equals(session.getDN())) {
                return session;
            } else {
                log.warn(
                        "DN ({}) was not the same with the DN ({}) in the session store!",
                        dn, session.getDN());
                return null;
            }
        }
    }

}
