/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.session.impl;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Timer;
import java.util.TimerTask;

import org.glite.pseudo.server.session.Session;
import org.glite.pseudo.server.session.StorageService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class implements the <code>StorageService</code> by storing the session
 * into the memory.
 */
public class InMemoryStorageService implements StorageService {

    /** Logging */
    private final Logger log = LoggerFactory
            .getLogger(InMemoryStorageService.class);

    /** All the stored sessions */
    private Hashtable<String, Session> sessions;

    /**
     * Constructs a new <code>InMemoryStorageService</code>
     * 
     * @param timer
     *            the <code>Timer</code> used in cleaning the expired
     *            <code>Session</code>s
     * @param frequency
     *            how often the timer should be run
     */
    public InMemoryStorageService(Timer timer, long frequency) {
        log.info(
                "Building a new Storage Service with sweeping frequency of {}",
                frequency);
        this.sessions = new Hashtable<String, Session>();
        timer.schedule(new MemorySessionsCleaningTask(), 0, frequency);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.session.StorageService#put(org.glite.pseudo.server
     * .session.Session)
     */
    public void put(Session session) {
        log.debug("Putting a session {} to the storage", session.getToken());
        sessions.put(session.getToken(), session);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.session.StorageService#remove(java.lang.String)
     */
    public void remove(String id) {
        if (sessions.containsKey(id)) {
            log.debug("Removing session {}", id);
            sessions.remove(sessions.get(id));
        } else {
            log.debug("Session {} not found to be removed", id);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.server.session.StorageService#get(java.lang.String)
     */
    public Session get(String id) {
        if (sessions.containsKey(id)) {
            return sessions.get(id);
        } else {
            log.debug("Session {} not found to be returned", id);
            return null;
        }
    }

    /**
     * This embedded class is used for cleaning the expired <code>Session</code>
     * s from the storage.
     */
    protected class MemorySessionsCleaningTask extends TimerTask {

        @Override
        public void run() {
            log.debug("Running the cleaning task for the expired sessions");
            Enumeration<Session> enumeration = sessions.elements();
            while (enumeration.hasMoreElements()) {
                Session session = enumeration.nextElement();
                if (!session.isValid()) {
                    log.debug("Cleaning the session {} from the storage",
                            session.getToken());
                    sessions.remove(session.getToken());
                }
            }
        }

    }
}
