/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.servlet;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.security.cert.X509Certificate;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.glite.pseudo.common.pki.CertificateExtension;
import org.glite.pseudo.server.Constants;
import org.glite.pseudo.server.attribute.Attribute;
import org.glite.pseudo.server.attribute.AttributeDefinitions;
import org.glite.pseudo.server.auditor.AuditorException;
import org.glite.pseudo.server.auditor.event.AuditEvent;
import org.glite.pseudo.server.auditor.event.PseudoAuditEvent;
import org.glite.pseudo.server.dn.DNBuilder;
import org.glite.pseudo.server.session.Session;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import eu.emi.security.authn.x509.proxy.ProxyUtils;

/**
 * This servlet serves the initial end-point for the users, and constructs the
 * CertInfoResponse -message for the clients, used for generating the CSR.
 */
public class LoginServlet extends AbstractServlet {

    /** The serial number */
    private static final long serialVersionUID = 2767774493143245316L;

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(LoginServlet.class);

    /** DNBuilder */
    private DNBuilder dnBuilder;

    /** AttributeDefinitions */
    private AttributeDefinitions attributeDefinitions;

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.server.servlet.AbstractServlet#init(javax.servlet.
     * ServletConfig)
     */
    public void init(ServletConfig config) throws ServletException {
        super.init(config);
        ServletContext context = config.getServletContext();
        dnBuilder = HttpServletHelper.getDNBuilder(context);
        attributeDefinitions = HttpServletHelper
                .getAttributeDefinitions(context);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.slcs.servlet.LoginServlet#doProcess(javax.servlet.http.
     * HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    protected void doProcess(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException {
        log.debug("doProcess...");

        // Stores the user attributes to a list
        List<Attribute> userAttributes = new Vector<Attribute>();
        // remote address of the user
        Attribute remoteAddress = new Attribute();
        remoteAddress.setName(Constants.USER_REMOTE_ADDRESS_ATTR_IDENTIFIER);
        remoteAddress.setValue(request.getRemoteAddr());
        userAttributes.add(remoteAddress);
        // user browser
        Attribute userAgent = getUserAgentAttribute(request);
        userAttributes.add(userAgent);
        // user certificate's subject DN
        Attribute userDN = null;

        try {
            // parsing the certficate chain without validation, as it is already
            // done by trustmanager
            X509Certificate[] clientCerts = null;
            clientCerts = (X509Certificate[]) request
                    .getAttribute("javax.servlet.request.X509Certificate");
            if (clientCerts == null || clientCerts.length == 0) {
                throw new IOException(
                        "No certificate chain found from the HttpServletRequest!");
            }
            if (log.isDebugEnabled()) {
                log.debug("The certificate chain length: " + clientCerts.length);
                for (int i = 0; i < clientCerts.length; i++) {
                    log.debug("Certificate #" + i + " DN: "
                            + clientCerts[i].getSubjectDN().toString());
                }
            }

            // store the client certificate subject DN to the attributes
            userDN = new Attribute();
            userDN.setName(Constants.USER_SUBJECT_DN_ATTR_IDENTIFIER);
            userDN.setValue(ProxyUtils.getEndUserCertificate(clientCerts).getSubjectDN().toString());
            userAttributes.add(userDN);

            // create a new DN
            String dn = dnBuilder.createDN(userAttributes);

            // store the new DN in sessions and get authorization token
            // SLCSSessions sessions = getSLCSSessions();
            Session session = sessionManager.createSession(dn);

            // store attributes in session
            session.setAttributes(userAttributes);

            // store the dn issuance event to the database
            try {
                AuditEvent dnIssue = new PseudoAuditEvent(
                        Constants.AUDIT_LEVEL_PSEUDO_DN_ISSUANCE, dn,
                        userAttributes);
                auditor.logEvent(dnIssue);
            } catch (AuditorException audit) {
                log.error("Audit error: ", audit);
            }

            log.debug("Session created: " + session);
            String authToken = session.getToken();
            String reqUrl = super.getContextUrl(request, "/certificate");

            List<CertificateExtension> extensions = certificatePolicy
                    .getRequiredCertificateExtensions(userAttributes);

            // send the login response back to the client
            sendLoginResponse(request, response, authToken, reqUrl, dn,
                    extensions);

        } catch (Exception e) {
            log.error("Sending error message to the client", e);
            // send the error response back to the client
            super.sendXMLErrorResponse(request, response, "CertInfoResponse",
                    e.getMessage(), e);
        }
    }

    /**
     * Returns the User-Agent header as Attribute.
     * 
     * @param req
     *            The HttpServletRequest object
     * @return The User-Agent attribute or <code>null</code> if not set.
     */
    protected Attribute getUserAgentAttribute(HttpServletRequest req) {
        String userAgent = req.getHeader("User-Agent");
        if (userAgent != null) {
            Attribute userAgentAttribute = attributeDefinitions
                    .createAttribute("UserAgent", userAgent);
            return userAgentAttribute;
        } else {
            return null;
        }
    }

    /**
     * Sends a XML CertInfoResponse back to the client.
     * 
     * <pre>
     *     &lt;CertInfoResponse&gt;
     *        &lt;Status&gt;Success&lt;/Status&gt;
     *        &lt;AuthorizationToken&gt;401B4A42F472565E84194BA03C0854B5DF44D22E8F34046810D605A03FAB8D85&lt;/AuthorizationToken&gt;
     *        &lt;CertificateRequest url=&quot;https://hestia.switch.ch:443/SLCS/certificate&quot;&gt;
     *           &lt;Subject&gt;CN=Tschopp Valery 9FEE5EE3,O=Switch - Teleinformatikdienste fuer Lehre und Forschung,C=CH&lt;/Subject&gt;
     *           &lt;CertificateExtension name=&quot;CertificatePolicies&quot; oid=&quot;2.5.29.32&quot; critical=&quot;false&quot;&gt;2.16.756.1.2.6.3&lt;/CertificateExtension&gt;
     *           &lt;CertificateExtension name=&quot;ExtendedKeyUsage&quot; oid=&quot;2.5.29.37&quot; critical=&quot;false&quot;&gt;ClientAuth&lt;/CertificateExtension&gt;
     *           &lt;CertificateExtension name=&quot;KeyUsage&quot; oid=&quot;2.5.29.15&quot; critical=&quot;true&quot;&gt;DigitalSignature,KeyEncipherment&lt;/CertificateExtension&gt;
     *           &lt;CertificateExtension name=&quot;SubjectAltName&quot; oid=&quot;2.5.29.17&quot; critical=&quot;false&quot;&gt;email:tschopp@switch.ch&lt;/CertificateExtension&gt;
     *        &lt;/CertificateRequest&gt;
     *     &lt;/CertInfoResponse&gt;
     * </pre>
     * 
     * @param req
     *            The HttpServletRequest object
     * @param res
     *            The HttpServletResponse object
     * @param authToken
     *            The authorization token
     * @param certDN
     *            The certificate subject (DN)
     * @param certExtensions
     *            List of CertificateExtension required by the CertificatePolicy
     * @throws IOException
     *             If an error occurs while writing the response.
     */
    protected void sendLoginResponse(HttpServletRequest req,
            HttpServletResponse res, String authToken, String requestURL,
            String certDN, List<CertificateExtension> certExtensions)
            throws IOException, ServletException {
        // build response
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        pw.println(getXMLDeclaration());

        // send response
        pw.println("<CertInfoResponse>");
        pw.println("<Status>Success</Status>");
        pw.print("<AuthorizationToken>");
        pw.print(authToken);
        pw.println("</AuthorizationToken>");
        // request URL
        pw.print("<ServiceEndpoint>");
        pw.print(requestURL);
        pw.println("</ServiceEndpoint>");
        pw.print("<SubjectDN>");
        pw.print(certDN);
        pw.println("</SubjectDN>");
        pw.println("<CertificatePolicy>");

        if (certExtensions != null && !certExtensions.isEmpty()) {
            log.debug("Certificate extensions found, adding them to the CertInfoResponse..");
            // add certificate extensions
            Iterator<CertificateExtension> extensions = certExtensions
                    .iterator();
            while (extensions.hasNext()) {
                CertificateExtension extension = extensions.next();
                log.debug("Extension added: {}", extension.toXML());
                pw.println(extension.toXML());
            }
        }
        pw.println("</CertificatePolicy>");
        pw.println("</CertInfoResponse>");
        if (log.isDebugEnabled()) {
            log.debug("sending CertInfoResponse:\n" + sw.getBuffer().toString());
        }
        // write response back to client
        res.setContentType("text/xml");
        PrintWriter out = res.getWriter();
        out.println(sw.getBuffer().toString());
        out.close();
    }

}
