/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.servlet;

import javax.servlet.ServletContext;

import org.glite.pseudo.common.util.Util;
import org.glite.pseudo.server.aaclient.AAClient;
import org.glite.pseudo.server.acl.AccessControlList;
import org.glite.pseudo.server.attribute.AttributeDefinitions;
import org.glite.pseudo.server.auditor.AuditAdmin;
import org.glite.pseudo.server.auditor.Auditor;
import org.glite.pseudo.server.caclient.CAClient;
import org.glite.pseudo.server.dn.DNBuilder;
import org.glite.pseudo.server.policy.CertificatePolicy;
import org.glite.pseudo.server.session.SessionManager;
import org.glite.pseudo.server.session.StorageService;

/**
 * A helper class that provides access to internal state from Servlets and hence
 * also JSPs.
 * 
 * This class is based on
 * <code>edu.internet2.middleware.shibboleth.idp.util.HttpServletHelper</code>
 * */
public class HttpServletHelper {

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link SessionManager} service: {@value} .
     */
    public static final String SESSION_MNGR_SID_CTX_PARAM = "SessionManagerId";

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link StorageService} service: {@value} .
     */
    public static final String STORAGE_SERVICE_SID_CTX_PARAM = "StorageServiceId";

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link Auditor} service: {@value} .
     */
    public static final String AUDITOR_SID_CTX_PARAM = "AuditorId";

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link AuditAdmin} service: {@value} .
     */
    public static final String AUDIT_ADMIN_SID_CTX_PARAM = "AuditAdminId";

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link CAClient} service: {@value} .
     */
    public static final String CA_CLIENT_SID_CTX_PARAM = "CAClientId";

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link AAClient} service: {@value} .
     */
    public static final String AA_CLIENT_SID_CTX_PARAM = "AAClientId";

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link DNBuilder} service: {@value} .
     */
    public static final String DN_BUILDER_SID_CTX_PARAM = "DNBuilderId";

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link CertificatePolicy} service: {@value} .
     */
    public static final String CERTIFICATE_POLICY_SID_CTX_PARAM = "CertificatePolicyId";

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link AttributeDefinitions} service: {@value} .
     */
    public static final String ATTRIBUTE_DEFINITIONS_SID_CTX_PARAM = "AttributeDefinitionsId";

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link AccessControlList} for administrator ACL service: {@value} .
     */
    public static final String ADMINISTRATOR_ACL_SID_CTX_PARAM = "AdministratorACL";

    /**
     * {@link ServletContext} parameter name bearing the ID of the
     * {@link AccessControlList} service: {@value} .
     */
    public static final String ACL_SID_CTX_PARAM = "ACL";

    /**
     * Default ID by which the {@link SessionManager} is know within the Servlet
     * context: {@value} .
     */
    public static final String DEFAULT_SESSION_MNGR_SID = "pseudonymity.SessionManager";

    /**
     * Default ID by which the {@link StorageService} is know within the Servlet
     * context: {@value} .
     */
    public static final String DEFAULT_STORAGE_SERVICE_SID = "pseudonymity.StorageService";

    /**
     * Default ID by which the {@link Auditor} is know within the Servlet
     * context: {@value} .
     */
    public static final String DEFAULT_AUDITOR_SID = "pseudonymity.Auditor";

    /**
     * Default ID by which the {@link AuditAdmin} is know within the Servlet
     * context: {@value} .
     */
    public static final String DEFAULT_AUDIT_ADMIN_SID = "pseudonymity.AuditAdmin";

    /**
     * Default ID by which the {@link CAClient} is know within the Servlet
     * context: {@value} .
     */
    public static final String DEFAULT_CA_CLIENT_SID = "pseudonymity.CAClient";

    /**
     * Default ID by which the {@link AAClient} is know within the Servlet
     * context: {@value} .
     */
    public static final String DEFAULT_AA_CLIENT_SID = "pseudonymity.AAClient";

    /**
     * Default ID by which the {@link DNBuilder} is know within the Servlet
     * context: {@value} .
     */
    public static final String DEFAULT_DN_BUILDER_SID = "pseudonymity.DNBuilder";

    /**
     * Default ID by which the {@link CertificatePolicy} is know within the
     * Servlet context: {@value} .
     */
    public static final String DEFAULT_CERTIFICATE_POLICY_SID = "pseudonymity.CertificatePolicy";

    /**
     * Default ID by which the {@link AttributeDefinitions} is know within the
     * Servlet context: {@value} .
     */
    public static final String DEFAULT_ATTRIBUTE_DEFINITIONS_SID = "pseudonymity.AttributeDefinitions";

    /**
     * Default ID by which the {@link AccessControlList} is know within the
     * Servlet context: {@value} .
     */
    public static final String DEFAULT_ACL_SID = "pseudonymity.ACL";

    /**
     * Default ID by which the {@link AccessControlList} for administrator ACL
     * is know within the Servlet context: {@value} .
     */
    public static final String DEFAULT_ADMINISTRATOR_ACL_SID = "pseudonymity.AdministratorACL";

    /**
     * Gets a value for a given context parameter. If no value is present the
     * default value is used.
     * 
     * @param context
     *            the Servlet context
     * @param name
     *            name of the context parameter
     * @param defaultValue
     *            default value of the parameter
     * 
     * @return the value of the context parameter or the default value if the
     *         parameter is not set or does not contain a value
     */
    public static String getContextParam(ServletContext context, String name,
            String defaultValue) {
        String value = Util
                .safeTrimOrNullString(context.getInitParameter(name));
        if (value == null) {
            value = defaultValue;
        }
        return value;
    }

    /**
     * Gets the {@link SessionManager} service bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static SessionManager getSessionManager(ServletContext context) {
        return getSessionManager(
                context,
                getContextParam(context, SESSION_MNGR_SID_CTX_PARAM,
                        DEFAULT_SESSION_MNGR_SID));
    }

    /**
     * Gets the {@link SessionManager} bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static SessionManager getSessionManager(ServletContext context,
            String serviceId) {
        return (SessionManager) context.getAttribute(serviceId);
    }

    /**
     * Gets the {@link StorageService} service bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static StorageService getStorageService(ServletContext context) {
        return getStorageService(
                context,
                getContextParam(context, STORAGE_SERVICE_SID_CTX_PARAM,
                        DEFAULT_STORAGE_SERVICE_SID));
    }

    /**
     * Gets the {@link StorageService} bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static StorageService getStorageService(ServletContext context,
            String serviceId) {
        return (StorageService) context.getAttribute(serviceId);
    }

    /**
     * Gets the {@link Auditor} service bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static Auditor getAuditor(ServletContext context) {
        return getAuditor(
                context,
                getContextParam(context, AUDITOR_SID_CTX_PARAM,
                        DEFAULT_AUDITOR_SID));
    }

    /**
     * Gets the {@link Auditor} bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static Auditor getAuditor(ServletContext context, String serviceId) {
        return (Auditor) context.getAttribute(serviceId);
    }

    /**
     * Gets the {@link AuditAdmin} service bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static AuditAdmin getAuditAdmin(ServletContext context) {
        return getAuditAdmin(
                context,
                getContextParam(context, AUDIT_ADMIN_SID_CTX_PARAM,
                        DEFAULT_AUDIT_ADMIN_SID));
    }

    /**
     * Gets the {@link AuditAdmin} bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static AuditAdmin getAuditAdmin(ServletContext context,
            String serviceId) {
        return (AuditAdmin) context.getAttribute(serviceId);
    }

    /**
     * Gets the {@link CAClient} service bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static CAClient getCAClient(ServletContext context) {
        return getCAClient(
                context,
                getContextParam(context, CA_CLIENT_SID_CTX_PARAM,
                        DEFAULT_CA_CLIENT_SID));
    }

    /**
     * Gets the {@link CAClient} bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static CAClient getCAClient(ServletContext context, String serviceId) {
        return (CAClient) context.getAttribute(serviceId);
    }

    /**
     * Gets the {@link AAClient} service bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static AAClient getAAClient(ServletContext context) {
        return getAAClient(
                context,
                getContextParam(context, AA_CLIENT_SID_CTX_PARAM,
                        DEFAULT_AA_CLIENT_SID));
    }

    /**
     * Gets the {@link AAClient} bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static AAClient getAAClient(ServletContext context, String serviceId) {
        return (AAClient) context.getAttribute(serviceId);
    }

    /**
     * Gets the {@link DNBuilder} service bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static DNBuilder getDNBuilder(ServletContext context) {
        return getDNBuilder(
                context,
                getContextParam(context, DN_BUILDER_SID_CTX_PARAM,
                        DEFAULT_DN_BUILDER_SID));
    }

    /**
     * Gets the {@link DNBuilder} bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static DNBuilder getDNBuilder(ServletContext context,
            String serviceId) {
        return (DNBuilder) context.getAttribute(serviceId);
    }

    /**
     * Gets the {@link CertificatePolicy} service bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static CertificatePolicy getCertificatePolicy(ServletContext context) {
        return getCertificatePolicy(
                context,
                getContextParam(context, CERTIFICATE_POLICY_SID_CTX_PARAM,
                        DEFAULT_CERTIFICATE_POLICY_SID));
    }

    /**
     * Gets the {@link CertificatePolicy} bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static CertificatePolicy getCertificatePolicy(
            ServletContext context, String serviceId) {
        return (CertificatePolicy) context.getAttribute(serviceId);
    }

    /**
     * Gets the {@link AttributeDefinitions} service bound to the Servlet
     * context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static AttributeDefinitions getAttributeDefinitions(
            ServletContext context) {
        return getAttributeDefinitions(
                context,
                getContextParam(context, ATTRIBUTE_DEFINITIONS_SID_CTX_PARAM,
                        DEFAULT_ATTRIBUTE_DEFINITIONS_SID));
    }

    /**
     * Gets the {@link AttributeDefinitions} bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static AttributeDefinitions getAttributeDefinitions(
            ServletContext context, String serviceId) {
        return (AttributeDefinitions) context.getAttribute(serviceId);
    }

    /**
     * Gets the {@link AccessControlList} service bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static AccessControlList getAccessControlList(ServletContext context) {
        return getAccessControlList(context,
                getContextParam(context, ACL_SID_CTX_PARAM, DEFAULT_ACL_SID));
    }

    /**
     * Gets the {@link AccessControlList} bound to the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static AccessControlList getAccessControlList(
            ServletContext context, String serviceId) {
        return (AccessControlList) context.getAttribute(serviceId);
    }

    /**
     * Gets the {@link AccessControlList} for administrator ACL service bound to
     * the Servlet context.
     * 
     * @param context
     *            the Servlet context
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static AccessControlList getAdministratorAccessControlList(
            ServletContext context) {
        return getAdministratorAccessControlList(
                context,
                getContextParam(context, ADMINISTRATOR_ACL_SID_CTX_PARAM,
                        DEFAULT_ADMINISTRATOR_ACL_SID));
    }

    /**
     * Gets the {@link AccessControlList} for administrator ACL bound to the
     * Servlet context.
     * 
     * @param context
     *            the Servlet context
     * @param serviceId
     *            the ID under which the service bound
     * 
     * @return the service or null if there is no such service bound to the
     *         context
     */
    public static AccessControlList getAdministratorAccessControlList(
            ServletContext context, String serviceId) {
        return (AccessControlList) context.getAttribute(serviceId);
    }

}
