/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.servlet;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.security.Principal;
import java.security.cert.X509Certificate;
import java.util.List;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.glite.pseudo.common.pki.Certificate;
import org.glite.pseudo.common.pki.CertificateRequest;
import org.glite.pseudo.server.Constants;
import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.aaclient.AAClient;
import org.glite.pseudo.server.aaclient.AAClientException;
import org.glite.pseudo.server.attribute.Attribute;
import org.glite.pseudo.server.auditor.event.AuditEvent;
import org.glite.pseudo.server.auditor.event.PseudoAuditEvent;
import org.glite.pseudo.server.caclient.CAClient;
import org.glite.pseudo.server.caclient.CAClientException;
import org.glite.pseudo.server.caclient.CAConnection;
import org.glite.pseudo.server.caclient.CARequest;
import org.glite.pseudo.server.caclient.CAResponse;
import org.glite.pseudo.server.session.Session;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import eu.emi.security.authn.x509.proxy.ProxyUtils;

/**
 * This servlet serves the end-point for obtaining the CSRs from the users, and
 * orchestrates the process of issuing and registering the pseudonymous
 * certificate using online CA and AA.
 */
public class CertificateServlet extends AbstractServlet {

    /** The serial number */
    private static final long serialVersionUID = 2767774493123124316L;

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(CertificateServlet.class);

    /** Identifier for AuthorizationToken in the request */
    private static final String REQ_AUTHZ_TOKEN_IDENTIFIER = "AuthorizationToken";

    /** Identifier for CertificateSigningRequest in the request */
    private static final String REQ_CSR_IDENTIFIER = "CertificateSigningRequest";

    /** AAClient */
    private AAClient aaClient;

    /** CAClient */
    private CAClient caClient;

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.server.servlet.AbstractServlet#init(javax.servlet.
     * ServletConfig)
     */
    public void init(ServletConfig config) throws ServletException {
        super.init(config);
        ServletContext context = config.getServletContext();
        aaClient = HttpServletHelper.getAAClient(context);
        caClient = HttpServletHelper.getCAClient(context);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.slcs.servlet.CertificateServlet#doProcess(javax.servlet.http
     * .HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    protected void doProcess(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException {
        log.debug("doProcess...");
        try {
            // get parameters
            String authorizationToken = getCertificateRequestParameter(request,
                    REQ_AUTHZ_TOKEN_IDENTIFIER);
            String pemCertificateRequest = getCertificateRequestParameter(
                    request, REQ_CSR_IDENTIFIER);
            log.info("AuthorizationToken=" + authorizationToken);
            log.debug("CertificateSigningRequest={}", pemCertificateRequest);

            // decode PEM certificate signing request
            CertificateRequest certificateRequest = null;
            try {
                StringReader reader = new StringReader(pemCertificateRequest);
                log.debug("Initializing PEM reader");
                certificateRequest = CertificateRequest.readPEM(reader);
                log.debug("Successfully decoded CSR {}", certificateRequest);
                // } catch (GeneralSecurityException e) {
            } catch (Exception e) {
                log.error(
                        "Failed to rebuild the PEM CertificateSigningRequest",
                        e);
                throw new PseudoServerException(
                        "Failed to decode PEM CertificateSigningRequest", e);
            }

            // check SLCS sessions
            Principal principal = certificateRequest.getPrincipal();
            String certificateSubject = principal.getName();
            log.debug("Subject={}", certificateSubject);
            log.debug("check session...");
            Session session = sessionManager.getSession(certificateSubject,
                    authorizationToken);
            if (session != null && session.isValid()) {
                log.debug(session + " is valid");

                List<Attribute> userAttributes = session.getAttributes();
                AuditEvent newRequest = new PseudoAuditEvent(
                        Constants.AUDIT_LEVEL_CERTIFICATE_REQUEST,
                        certificateSubject, userAttributes);
                auditor.logEvent(newRequest);

                // check certificate against policy
                log.debug("check certificate request against policy...");
                if (certificatePolicy.isCertificateRequestValid(
                        certificateRequest, userAttributes)) {
                    Certificate cert = requestCertificate(certificateRequest,
                            principal);
                    if (cert != null) { // certificate obtained
                        // store the event
                        AuditEvent certIssued = new PseudoAuditEvent(
                                Constants.AUDIT_LEVEL_CERTIFICATE_ISSUANCE,
                                certificateSubject, userAttributes);
                        auditor.logEvent(certIssued);
                        // obtain the client certificate chain
                        X509Certificate[] clientCerts = null;
                        clientCerts = (X509Certificate[]) request
                                .getAttribute("javax.servlet.request.X509Certificate");
                        // register the certificate to the attribute authority
                        boolean registered = registerCertificate(clientCerts,
                                cert);
                        if (registered) { // certificate successfully registered
                            AuditEvent certRegistered = new PseudoAuditEvent(
                                    Constants.AUDIT_LEVEL_CERTIFICATE_REGISTRATION,
                                    certificateSubject, userAttributes);
                            auditor.logEvent(certRegistered);
                        }
                        // send response back to the client
                        log.debug("Sending the CertificateResponse back to the client");
                        sendXMLCerfificateResponse(request, response,
                                certificateSubject, cert);
                        log.info("Sent CertResponse back to the client.");

                    } else { // error during certificate request
                        // store the event
                        AuditEvent certError = new PseudoAuditEvent(
                                Constants.AUDIT_LEVEL_CERTIFICATE_ISSUANCE,
                                PseudoAuditEvent.LEVEL_ERROR,
                                certificateSubject, userAttributes);
                        auditor.logEvent(certError);
                    }
                } else { // policy was not valid
                    throw new PseudoServerException(
                            "CertificateSigningRequest is not conform to CertificatePolicy");
                }
            } else { // Session was not valid
                throw new PseudoServerException("Session: "
                        + authorizationToken + "," + certificateSubject
                        + " does not exists or is expired");
            }

        } catch (Exception e) {
            // any error during the certificate request or registration or
            // auditing
            log.error("", e);
            sendXMLErrorResponse(request, response, "CertResponse",
                    e.getMessage(), e);
        }
    }

    /**
     * Requests a certificate from online CA.
     * 
     * @param certificateRequest
     *            The certificate request
     * @param principal
     *            The certificate owner principal
     * @return The certificate signed by the online CA, null if any errors
     *         occured
     */
    protected Certificate requestCertificate(
            CertificateRequest certificateRequest, Principal principal) {
        Certificate result = null;
        try {
            log.debug("get CA connection");
            CAConnection connection = caClient.getConnection();
            log.debug("create CA request");
            CARequest csrRequest = connection.createRequest(certificateRequest);
            log.info("send certificate request to CA server");
            connection.sendRequest(csrRequest);
            log.info("read CA server response");
            CAResponse csrResponse = connection.getResponse();
            log.debug("get certificate");
            result = csrResponse.getCertificate(principal);
        } catch (CAClientException e) {
            log.error("Could not obtain certificate from the online CA!", e);
        }
        return result;
    }

    /**
     * Registers a certificate to the attribute authority.
     * 
     * @param clientCerts
     *            The certificate chain of the pseudonymity service user
     * @param cert
     *            The certificate to be registered
     * @return <code>true</code> if registration succeeded, <code>false</code>
     *         otherwise
     */
    protected boolean registerCertificate(X509Certificate[] clientCerts,
            Certificate cert) {
        boolean registered = false;
        try {
            if (aaClient != null) {
                aaClient.registerCertificate(ProxyUtils.getEndUserCertificate(clientCerts),
                        cert.getCertificate());
                log.debug("Certificate successfully registered.");
                registered = true;
            } else {
                log.debug("Could not register the certificate to AA, as the AAClient was null!");
            }
        } catch (AAClientException e) {
            log.error("Certificate registration to the AA failed!", e);
        }
        return registered;
    }

    /**
     * Checks if the CertificateRequest contains a no empty parameter and
     * returns the value.
     * 
     * @param request
     *            The HttpServletRequest object.
     * @param name
     *            The request parameter name to check.
     * @return The parameter value.
     * @throws PseudoServerException
     *             If the parameter is not present or if the value is empty.
     */
    protected String getCertificateRequestParameter(HttpServletRequest request,
            String name) throws PseudoServerException {
        String value = request.getParameter(name);
        if (value == null || value.equals("")) {
            log.error("Request parameter " + name + " is missing or empty.");
            throw new PseudoServerException("Request parameter " + name
                    + " is missing or empty");
        }
        return value;
    }

    /**
     * Sends a CertResponse back to the client.
     * 
     * @param req
     *            The HttpServerRequest object
     * @param res
     *            The HttpServletResponse object.
     * @param subject
     *            The certificate subject (DN).
     * @param certificate
     *            The Certificate object.
     * @throws IOException
     *             If an error occurs while sending the response.
     */
    protected void sendXMLCerfificateResponse(HttpServletRequest req,
            HttpServletResponse res, String dn, Certificate certificate)
            throws IOException {
        // build response
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        pw.println(getXMLDeclaration());
        pw.println("<CertResponse>");
        pw.println("<Status>Success</Status>");
        pw.println("<Subject>" + dn + "</Subject>");
        String pemCertificate = certificate.getPEM();
        pw.println("<Certificate format=\"PEM\">" + pemCertificate
                + "</Certificate>");
        pw.println("</CertResponse>");

        log.debug("CertResponse: {}", sw.getBuffer().toString());

        // write response back to client
        res.setContentType("text/xml");
        PrintWriter out = res.getWriter();
        out.println(sw.getBuffer().toString());
        out.close();

    }

}
