/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.servlet;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.security.Security;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.glite.pseudo.server.auditor.Auditor;
import org.glite.pseudo.server.policy.CertificatePolicy;
import org.glite.pseudo.server.session.SessionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * AbstractServlet is the base class for the servlets in Pseudonymity Service.
 * 
 */
public abstract class AbstractServlet extends HttpServlet {

    /** The serial number */
    private static final long serialVersionUID = 2767774490991442316L;

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(AbstractServlet.class);

    /** Auditor */
    protected Auditor auditor;

    /** SessionManager */
    protected SessionManager sessionManager;

    /** CertificatePolicy */
    protected CertificatePolicy certificatePolicy;

    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.GenericServlet#init(javax.servlet.ServletConfig)
     */
    public void init(ServletConfig config) throws ServletException {
        // add only once
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
        ServletContext context = config.getServletContext();
        sessionManager = HttpServletHelper.getSessionManager(context);
        auditor = HttpServletHelper.getAuditor(context);
        certificatePolicy = HttpServletHelper.getCertificatePolicy(context);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest
     * , javax.servlet.http.HttpServletResponse)
     */
    protected void doGet(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException {
        doProcess(request, response);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest
     * , javax.servlet.http.HttpServletResponse)
     */
    protected void doPost(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException {
        doProcess(request, response);
    }

    /**
     * Processes the POST or GET request.
     * 
     * @param request
     *            The HttpServletRequest object.
     * @param response
     *            The HttpServletRespose object.
     * @throws IOException
     * @throws ServletException
     */
    protected abstract void doProcess(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException;

    /**
     * Return the servlet URL for the servlet path. The servlet path must starts
     * with '/' and is relative the the context root.
     * 
     * @param req
     *            The HttpServletRequest object.
     * @param servletPath
     *            The servlet path relative to the context root.
     * @return The URL of the servlet.
     */
    protected String getContextUrl(HttpServletRequest req, String servletPath) {
        StringBuffer sb = new StringBuffer();
        sb.append(req.getScheme()).append("://");
        sb.append(req.getServerName()).append(':').append(req.getServerPort());
        sb.append(req.getContextPath());
        sb.append(servletPath);
        return sb.toString();
    }

    /**
     * Sends a XML error message back to the client.
     * 
     * @param res
     *            The HttpServletResponse object.
     * @param type
     *            The type of the response.
     * @param message
     *            The error message.
     * @param cause
     *            The error cause.
     * @throws IOException
     *             If an error occurs while sending the error.
     */
    protected void sendXMLErrorResponse(HttpServletRequest req,
            HttpServletResponse res, String type, String message,
            Throwable cause) throws IOException {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        pw.println(getXMLDeclaration());
        pw.print("<");
        pw.print(type);
        pw.println(">");
        pw.println("<Status>Error</Status>");
        pw.print("<Error>");
        pw.print(message);
        pw.println("</Error>");
        if (cause != null) {
            pw.print("<StackTrace>");
            cause.printStackTrace(pw);
            pw.print("</StackTrace>");
        }
        pw.print("</");
        pw.print(type);
        pw.println(">");

        log.info("sending " + type + ":\n" + sw.getBuffer().toString());

        // send response
        res.setContentType("text/xml");
        PrintWriter out = res.getWriter();
        out.println(sw.getBuffer().toString());
        out.close();
    }

    /**
     * Returns the XML declaration header:
     * 
     * <pre>
     *      &lt;?xml version=&quot;1.0&quot; encoding=&quot;UTF-8&quot; ?&gt;
     * </pre>
     * 
     * @return The XML declation header.
     */
    protected String getXMLDeclaration() {
        return "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>";
    }
}
