/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.policy.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.glite.pseudo.common.pki.CertificateExtension;
import org.glite.pseudo.common.pki.CertificateExtensionFactory;
import org.glite.pseudo.common.pki.CertificateRequest;
import org.glite.pseudo.server.attribute.Attribute;
import org.glite.pseudo.server.config.PseudoServerConfiguration;
import org.glite.pseudo.server.policy.CertificatePolicy;
import org.ini4j.Ini;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * MandatoryCertificateExtensionsPolicy implements a {@link CertificatePolicy}
 * where all defined {@link CertificateExtension}s are mandatory and exclusive.
 * 
 * <p>
 * In the <code>&lt;CertificateExtension&gt;</code> element, the value of the
 * extension can use variable in the form <code>${attribute-name}</code> to have
 * user dependent value resolved dynamically.
 * 
 * <pre>
 * [CertificatePolicy]
 * implementation = org.glite.pseudo.server.policy.impl.MandatoryCertificateExtensionsPolicy
 * extensionId = KeyUsage
 * extensionCritical = true
 * extensionValue = DigitalSignature,KeyEncipherment
 * extensionId = SubjectAltName
 * extensionCritical = false
 * extensionValue = email:${email}
 * extensionId = CertificatePolicies
 * extensionCritical = false
 * extensionValue = 2.16.756.1.2.6.4.1.0
 * extensionId = ExtendedKeyUsage
 * extensionCritical = false
 * extensionValue = ClientAuth
 * </pre>
 */
public class MandatoryCertificateExtensionsPolicy implements CertificatePolicy {

    /** Logging */
    private final Logger log = LoggerFactory
            .getLogger(MandatoryCertificateExtensionsPolicy.class);

    /** Map of name-values certificate extension required by the policy */
    private Map<String, String> requiredCertificateExtensionValuesMap_ = null;

    /** Map of the name-boolean if certificate extension is critical or not */
    private Map<String, Boolean> requiredCertificateExtensionCriticalMap_ = null;

    /**
     * Constructs a new <code>MandatoryCertificateExtensionsPolicy</code>.
     */
    public MandatoryCertificateExtensionsPolicy() {
        // no op
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.policy.CertificatePolicy#isCertificateRequestValid
     * (org.glite.pseudo.common.pki.CertificateRequest, java.util.List)
     */
    public boolean isCertificateRequestValid(CertificateRequest request,
            List<Attribute> attributes) {
        // check validity of extensions
        List<CertificateExtension> requiredCertificateExtensions = getRequiredCertificateExtensions(attributes);
        List<CertificateExtension> certificateRequestExtensions = request
                .getCertificateExtensions();
        if (!certificateRequestExtensions
                .containsAll(requiredCertificateExtensions)) {
            log.error("CertificateSigningRequest does not contain all required CertificateExtensions");
            return false;
        } else if (!requiredCertificateExtensions
                .containsAll(certificateRequestExtensions)) {
            log.error("CertificateSigningRequest contains more CertificateExtensions than required");
            return false;
        }
        return true;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.server.policy.CertificatePolicy#
     * getRequiredCertificateExtensions(java.util.List)
     */
    public List<CertificateExtension> getRequiredCertificateExtensions(
            List<Attribute> attributes) {
        // convert List to Map
        Map<String, String> attributesMap = new HashMap<String, String>();
        Iterator<Attribute> attributesIter = attributes.iterator();
        while (attributesIter.hasNext()) {
            Attribute attribute = attributesIter.next();
            String name = attribute.getName();
            String value = attribute.getValue();
            if (attributesMap.containsKey(name)) {
                // aggregate multi-valued attributes with ;
                String oldValue = (String) attributesMap.get(name);
                value = oldValue + ";" + value;
            }
            attributesMap.put(name, value);
        }
        return getRequiredCertificateExtensions(attributesMap);
    }

    /**
     * Returns a list of required {@link CertificateExtension}s, already
     * evaluated with the corresponding user attribute values.
     * 
     * @param attributes
     *            Map of user attributes.
     * @return List of {@link CertificateExtension}s
     */
    private List<CertificateExtension> getRequiredCertificateExtensions(
            Map<String, String> attributes) {
        List<CertificateExtension> requiredCertificateExtensions = new ArrayList<CertificateExtension>();
        Iterator<String> extensionNames = requiredCertificateExtensionValuesMap_
                .keySet().iterator();
        while (extensionNames.hasNext()) {
            String extensionName = (String) extensionNames.next();
            String extensionValues = (String) requiredCertificateExtensionValuesMap_
                    .get(extensionName);
            log.debug("Processing extension name={}, values={}", extensionName,
                    extensionValues);
            log.debug("Extension should be critical: {}",
                    this.requiredCertificateExtensionCriticalMap_
                            .get(extensionName));
            boolean extensionCritical = this.requiredCertificateExtensionCriticalMap_
                    .get(extensionName).booleanValue();
            log.debug("Result: {}", extensionCritical);
            if (extensionValues.indexOf("${") != -1) {
                // values contains user attribute variable(s), substitute.
                Iterator<String> attributeNames = attributes.keySet()
                        .iterator();
                while (attributeNames.hasNext()) {
                    String attributeName = (String) attributeNames.next();
                    String placeholder = "${" + attributeName + "}";
                    if (extensionValues.indexOf(placeholder) != -1) {
                        String replace = "\\$\\{" + attributeName + "\\}";
                        String attributeValue = attributes.get(attributeName)
                                .toString();
                        if (log.isDebugEnabled()) {
                            log.debug("CertificateExtension values:"
                                    + extensionValues + " replace regex:"
                                    + replace + " by:" + attributeValue);
                        }
                        extensionValues = extensionValues.replaceAll(replace,
                                attributeValue);
                    }
                }
            }
            // create the extension with the values expanded.
            log.debug("Creating a new CertificateExtension object");
            CertificateExtension extension = CertificateExtensionFactory
                    .createCertificateExtension(extensionName, extensionValues,
                            extensionCritical);
            requiredCertificateExtensions.add(extension);
        }
        return requiredCertificateExtensions;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.PseudoServerComponent#init(org.glite.pseudo.server
     * .config.PseudoServerConfiguration)
     */
    public void init(PseudoServerConfiguration configuration) {
        log.debug("Initializing certificate policy");
        Ini.Section cfgSection = configuration
                .getCertificatePolicyConfiguration();
        String[] ids = cfgSection.getAll("extensionId", String[].class);
        String[] criticals = cfgSection.getAll("extensionCritical",
                String[].class);
        String[] values = cfgSection.getAll("extensionValue", String[].class);
        if (ids.length != criticals.length || criticals.length != values.length) {
            log.error("The configuration for CertificatePolicy is not correct: the amount of extensionId, extensionCritical and extensionValue parameters do not match!");
        } else {
            log.debug("{} extensions defined in the policy.", ids.length);
            this.requiredCertificateExtensionCriticalMap_ = new HashMap<String, Boolean>();
            this.requiredCertificateExtensionValuesMap_ = new HashMap<String, String>();
            for (int i = 0; i < ids.length; i++) {
                this.requiredCertificateExtensionValuesMap_.put(ids[i],
                        values[i]);
                this.requiredCertificateExtensionCriticalMap_.put(ids[i],
                        new Boolean(criticals[i]));
            }
            log.info("Certificate policy successfully initialized.");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.server.PseudoServerComponent#shutdown()
     */
    public void shutdown() {
        // TODO Auto-generated method stub

    }

}
