/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.filter;

import java.io.IOException;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Vector;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletResponse;

import org.glite.pseudo.server.acl.AccessControlList;
import org.glite.pseudo.server.attribute.Attribute;
import org.glite.pseudo.server.servlet.HttpServletHelper;
import org.glite.voms.VOMSAttribute;
import org.glite.voms.VOMSValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * VOMSAttributesAuthorizationFilter is an ACL filter based on VOMS attributes.
 * The filter uses the underlying AccessControlList implementation to check if
 * the user is authorized. The implementation is based on
 * org.glite.slcs.filter.AttributesAuthorizationFilter.
 * </p>
 * 
 * <p>
 * Example acl-file configuration for allowing users from VO 'test':
 * </p>
 * 
 * <pre>
 * &lt;AccessControlList&gt;
 *    &lt;AccessControlRule id=&quot;1&quot;group=&quot;testgroup&quot;&gt;
 *       &lt;Attribute name=&quot;VOMS-FQAN&quot;&gt;/test%lt;/Attribute&gt;
 *    &lt;/AccessControlRule&gt;
 * &lt;/AccessControlList&gt;
 * </pre>
 */
public class VOMSAttributesAuthorizationFilter implements Filter {

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(VOMSAttributesAuthorizationFilter.class);

    /** Attributes ACL */
    private AccessControlList m_accessControlList = null;

    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
     */
    public void init(FilterConfig filterConfig) throws ServletException {
        ServletContext context = filterConfig.getServletContext();
        m_accessControlList = HttpServletHelper.getAccessControlList(context);
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest,
     * javax.servlet.ServletResponse, javax.servlet.FilterChain)
     */
    public void doFilter(ServletRequest request, ServletResponse response,
            FilterChain chain) throws IOException, ServletException {
        log.debug("doFilter start");
        X509Certificate[] clientCerts = null;
        clientCerts = (X509Certificate[]) request
                .getAttribute("javax.servlet.request.X509Certificate");
        if (clientCerts == null || clientCerts.length == 0) {
            throw new ServletException(
                    "No certificate chain found from the HttpServletRequest!");
        }
        log.debug("The certificate chain length: {}", clientCerts.length);
        VOMSValidator voms = new VOMSValidator(clientCerts);
        voms.validate();
        @SuppressWarnings("unchecked")
        Vector<VOMSAttribute> vomsAttrs = (Vector<VOMSAttribute>) VOMSValidator
                .parse(clientCerts);
        log.debug("VOMS attributes amount: {}", vomsAttrs.size());
        List<Attribute> userAttributes = new Vector<Attribute>();
        for (int i = 0; i < vomsAttrs.size(); i++) {
            VOMSAttribute vomsAttr = (VOMSAttribute) vomsAttrs.elementAt(i);
            List<?> fqans = vomsAttr.getFullyQualifiedAttributes();
            for (int j = 0; j < fqans.size(); j++) {
                log.debug("VOMS attribute {}:{}", j, fqans.get(j));
                Attribute attribute = new Attribute();
                attribute.setName("VOMS-FQAN");
                attribute.setValue(fqans.get(j).toString());
                userAttributes.add(attribute);
            }
        }
        if (m_accessControlList.isAuthorized(userAttributes)) {
            chain.doFilter(request, response);
        } else {
            String remoteAddress = request.getRemoteAddr();
            log.error(HttpServletResponse.SC_UNAUTHORIZED + ": User (IP:"
                    + remoteAddress + ") is not authorized: " + userAttributes);
            HttpServletResponse httpResponse = (HttpServletResponse) response;
            httpResponse
                    .sendError(HttpServletResponse.SC_UNAUTHORIZED,
                            "Based on your VOMS attributes, you are not authorized to access this service");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.Filter#destroy()
     */
    public void destroy() {
        log.debug("Shutting down.");
        m_accessControlList.shutdown();
    }

}
