/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.filter;

import java.io.IOException;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Vector;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletResponse;

import org.glite.pseudo.server.acl.AccessControlList;
import org.glite.pseudo.server.attribute.Attribute;
import org.glite.pseudo.server.servlet.HttpServletHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import eu.emi.security.authn.x509.proxy.ProxyUtils;

/**
 * <p>
 * SubjectDNAuthorizationFilter is an ACL filter based on X.509 certificate
 * subject DNs. The filter uses the underlying AccessControlList implementation
 * to check if the user is authorized. The implementation is based on
 * org.glite.slcs.filter.AttributesAuthorizationFilter.
 * </p>
 * 
 * <p>
 * Example acl-file configuration for allowing user with the subject DN
 * 'CN=Test, DC=ch':
 * </p>
 * 
 * <pre>
 * &lt;AccessControlList&gt;
 *    &lt;AccessControlRule id="1" group="admins"&gt;
 *       &lt;Attribute name="Certificate-DN"&gt;CN=Test, DC=ch&lt;/Attribute&gt;
 *    &lt;/AccessControlRule&gt;
 * &lt;/AccessControlList&gt;
 * </pre>
 */
public class SubjectDNAuthorizationFilter implements Filter {

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(SubjectDNAuthorizationFilter.class);

    /** Attributes ACL */
    private AccessControlList m_accessControlList = null;

    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
     */
    public void init(FilterConfig filterConfig) throws ServletException {
        ServletContext context = filterConfig.getServletContext();
        this.m_accessControlList = HttpServletHelper
                .getAdministratorAccessControlList(context);
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest,
     * javax.servlet.ServletResponse, javax.servlet.FilterChain)
     */
    public void doFilter(ServletRequest request, ServletResponse response,
            FilterChain filterChain) throws IOException, ServletException {
        log.debug("doFilter start");
        X509Certificate[] clientCerts = null;
        clientCerts = (X509Certificate[]) request
                .getAttribute("javax.servlet.request.X509Certificate");
        if (clientCerts == null || clientCerts.length == 0) {
            throw new ServletException(
                    "No certificate chain found from the HttpServletRequest!");
        }
        log.debug("The certificate chain length: {}", clientCerts.length);
        X509Certificate endEntityCert = ProxyUtils.getEndUserCertificate(clientCerts);
        List<Attribute> userAttributes = new Vector<Attribute>(1);
        Attribute attribute = new Attribute();
        attribute.setName("Certificate-DN");
        attribute.setValue(endEntityCert.getSubjectDN().toString());
        userAttributes.add(attribute);
        if (m_accessControlList.isAuthorized(userAttributes)) {
            filterChain.doFilter(request, response);
        } else {
            String remoteAddress = request.getRemoteAddr();
            log.error(HttpServletResponse.SC_UNAUTHORIZED + ": User (IP:"
                    + remoteAddress + ") is not authorized: " + userAttributes);
            HttpServletResponse httpResponse = (HttpServletResponse) response;
            httpResponse
                    .sendError(HttpServletResponse.SC_UNAUTHORIZED,
                            "Based on your X.509 certificate, you are not authorized to access this page");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.Filter#destroy()
     */
    public void destroy() {
        log.debug("Shutting down.");
        this.m_accessControlList.shutdown();
    }
}
