/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.dn.impl;

import java.security.SecureRandom;
import java.util.List;

import org.apache.commons.ssl.util.Hex;
import org.glite.pseudo.common.util.Util;
import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.attribute.Attribute;
import org.glite.pseudo.server.config.PseudoServerConfiguration;
import org.glite.pseudo.server.dn.DNBuilder;
import org.ini4j.Ini;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * Simple DNBuilder implementation for building pseudonymous DNs.
 * </p>
 */
public class PseudoDNBuilder implements DNBuilder {

    /** Field in the DN template to be replaced with the pseudonymous identifier */
    public static final String PSEUDO_IDENTIFIER_REGEX = "\\$\\{pseudoIdentifier\\}";

    /** Configuration identifier for DN template */
    public static final String DN_TEMPLATE_IDENTIFIER = "DNTemplate";

    /** Configuration identifier for the pseudonymoys identifier length */
    public static final String ID_LENGTH_IDENTIFIER = "IDLength";

    /** Logging */
    private final Logger log = LoggerFactory.getLogger(PseudoDNBuilder.class);

    /** Pattern for the DN */
    private String m_pattern;

    /** How many random characters to be included in the pseudonym */
    private int m_idLength;

    /** random bytes generator */
    private SecureRandom m_random = new SecureRandom();

    /**
     * Constructs a new <code>PseudoDNBuilder</code>.
     */
    public PseudoDNBuilder() {
        // no op
    }

    /**
     * <p>
     * Creates a pseudonymous DN according to the configuration defined in the
     * instance variables.
     * </p>
     * 
     * @return The pseudonymous DN as String
     */
    public String createDN(List<Attribute> attributes) {
        log.debug("Generating pseudo DN, idLength={}", this.getIdLength());
        String pseudoId = createRandomString(this.getIdLength());
        log.debug("Generated identifier: {}", pseudoId);
        String dnStr = this.getPattern();
        dnStr = dnStr.replaceAll(PSEUDO_IDENTIFIER_REGEX, pseudoId);
        log.debug("Generated DN: {}", dnStr);
        return dnStr;
    }

    /**
     * <p>
     * Gets the pseudonymous DN pattern.
     * </p>
     * 
     * @return the DN pattern
     */
    private String getPattern() {
        return this.m_pattern;
    }

    /**
     * <p>
     * Gets the pseudonym length.
     * </p>
     * 
     * @return the length of the pseudonym
     */
    private int getIdLength() {
        return this.m_idLength;
    }

    /**
     * <p>
     * Creates a String containing a specific amount of random characters.
     * </p>
     * 
     * @param length
     *            the length of the String
     * @return the random String
     */
    private String createRandomString(int length) {
        byte[] randomBytes = null;
        randomBytes = new byte[length];
        m_random.nextBytes(randomBytes);
        return Hex.encode(randomBytes);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.PseudoServerComponent#init(org.glite.pseudo.server
     * .config.PseudoServerConfiguration)
     */
    public void init(PseudoServerConfiguration configuration)
            throws PseudoServerException {
        Ini.Section cfgSection = configuration.getDNBuilderConfiguration();
        String dnPattern = Util.safeTrimOrNullString(cfgSection
                .get(DN_TEMPLATE_IDENTIFIER));
        if (dnPattern == null) {
            throw new PseudoServerException("DN Pattern cannot be empty!");
        }
        String pseudoIdLength = Util.safeTrimOrNullString(cfgSection
                .get(ID_LENGTH_IDENTIFIER));
        int length;
        if (pseudoIdLength == null) {
            throw new PseudoServerException(
                    "Pseudonymous identifier length cannot be null!");
        } else {
            length = Integer.valueOf(pseudoIdLength).intValue();
        }
        log.debug(
                "Initializing a pseudonymous DN builder with pattern={} and idLength={}",
                dnPattern, length);
        this.m_idLength = length;
        this.m_pattern = dnPattern;

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.server.PseudoServerComponent#shutdown()
     */
    public void shutdown() {
        // no op
    }

}
