/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.config;

import java.io.File;
import java.io.IOException;

import org.ini4j.Ini;
import org.ini4j.InvalidFileFormatException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PseudoServerConfiguration extends Ini {

    /** The serial number */
    private static final long serialVersionUID = 2767774490991448133L;

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(PseudoServerConfiguration.class);

    /** The name of the configuration (INI) section for CA client */
    protected static final String CFG_SECTION_CACLIENT = "CAClient";

    /** The name of the configuration (INI) section for AA client */
    protected static final String CFG_SECTION_AACLIENT = "AAClient";

    /** The name of the configuration (INI) section for certificate policy */
    protected static final String CFG_SECTION_CERTIFICATE_POLICY = "CertificatePolicy";

    /** The name of the configuration (INI) section for DN builder */
    protected static final String CFG_SECTION_DN_BUILDER = "DNBuilder";

    /** The name of the configuration (INI) section for auditor */
    protected static final String CFG_SECTION_AUDITOR = "Auditor";

    /** The name of the configuration (INI) section for audit administrator */
    protected static final String CFG_SECTION_AUDIT_ADMIN = "AuditAdmin";

    /** The name of the configuration (INI) section for attribute definitions */
    protected static final String CFG_SECTION_ATTRIBUTE_DEFINITIONS = "AttributeDefinitions";

    /** The name of the configuration (INI) section for access control list */
    protected static final String CFG_SECTION_ACL = "ACL";

    /** The name of the configuration (INI) section for administrator ACL */
    protected static final String CFG_SECTION_ADMINISTRATOR_ACL = "AdministratorACL";

    /** The configuration variable for the name of the implementing class */
    public static final String CFG_IMPLEMENTATION_ID = "implementation";

    /**
     * Constructs a new <code>PseudoServerConfiguration</code>.
     * 
     * @param iniFile
     *            the full path of the configuration (INI) file
     * @throws InvalidFileFormatException
     *             if the configuration file is invalid
     * @throws IOException
     *             if the configuration file cannot be read
     */
    public PseudoServerConfiguration(String iniFile)
            throws InvalidFileFormatException, IOException {
        super(new File(iniFile));
        log.info(
                "Successfully constructed a new configuration from the file {}",
                iniFile);
    }

    /**
     * Gets the configuration section for the CA client
     * 
     * @return related configuration section
     */
    public Ini.Section getCAClientConfiguration() {
        return this.get(CFG_SECTION_CACLIENT);
    }

    /**
     * Gets the configuration section for the AA client
     * 
     * @return related configuration section
     */
    public Ini.Section getAAClientConfiguration() {
        return this.get(CFG_SECTION_AACLIENT);
    }

    /**
     * Gets the configuration section for the certificate policy
     * 
     * @return related configuration section
     */
    public Ini.Section getCertificatePolicyConfiguration() {
        return this.get(CFG_SECTION_CERTIFICATE_POLICY);
    }

    /**
     * Gets the configuration section for the DN builder
     * 
     * @return related configuration section
     */
    public Ini.Section getDNBuilderConfiguration() {
        return this.get(CFG_SECTION_DN_BUILDER);
    }

    /**
     * Gets the configuration section for the auditor
     * 
     * @return related configuration section
     */
    public Ini.Section getAuditorConfiguration() {
        return this.get(CFG_SECTION_AUDITOR);
    }

    /**
     * Gets the configuration section for the audit administrator
     * 
     * @return related configuration section
     */
    public Ini.Section getAuditAdminConfiguration() {
        return this.get(CFG_SECTION_AUDIT_ADMIN);
    }

    /**
     * Gets the configuration section for the attribute definitions
     * 
     * @return related configuration section
     */
    public Ini.Section getAttributeDefinitionsConfiguration() {
        return this.get(CFG_SECTION_ATTRIBUTE_DEFINITIONS);
    }

    /**
     * Gets the configuration section for the access control list
     * 
     * @return related configuration section
     */
    public Ini.Section getAccessControlListConfiguration() {
        return this.get(CFG_SECTION_ACL);
    }

    /**
     * Gets the configuration section for the administrator ACL
     * 
     * @return related configuration section
     */
    public Ini.Section getAdminAccessControlListConfiguration() {
        return this.get(CFG_SECTION_ADMINISTRATOR_ACL);
    }

}
