/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.config;

import org.glite.pseudo.server.PseudoServerComponent;
import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.aaclient.AAClient;
import org.glite.pseudo.server.acl.AccessControlList;
import org.glite.pseudo.server.attribute.AttributeDefinitions;
import org.glite.pseudo.server.auditor.AuditAdmin;
import org.glite.pseudo.server.auditor.Auditor;
import org.glite.pseudo.server.caclient.CAClient;
import org.glite.pseudo.server.dn.DNBuilder;
import org.glite.pseudo.server.policy.CertificatePolicy;
import org.ini4j.Ini;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is used for constructing the configured implementation classes for
 * the Pseudonymity Service interfaces.
 */
public class ComponentFactory {

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(ComponentFactory.class);

    /** The configuration */
    private PseudoServerConfiguration configuration;

    /**
     * Constructs a new <code>ComponentFactory</code>
     * 
     * @param configuration
     *            for the Pseudonymity Service
     */
    public ComponentFactory(PseudoServerConfiguration configuration) {
        log.debug("Initializing a ComponentFactory");
        this.configuration = configuration;
    }

    /**
     * Creates an instance of the <code>AttributeDefinitions</code>
     * implementation.
     * 
     * @return the configured implementation
     */
    public AttributeDefinitions createAttributeDefinitionsInstance() {
        log.debug("Creating a new AttributeDefinitions instance");
        String className = getImplementationClassName(this.configuration
                .getAttributeDefinitionsConfiguration());
        return (AttributeDefinitions) this.createComponentInstance(className);
    }

    /**
     * Creates an instance of the <code>CertificatePolicy</code> implementation.
     * 
     * @return the configured implementation
     */
    public CertificatePolicy createCertificatePolicyInstance() {
        log.debug("Creating a new CertificatePolicy instance");
        String className = getImplementationClassName(this.configuration
                .getCertificatePolicyConfiguration());
        return (CertificatePolicy) this.createComponentInstance(className);
    }

    /**
     * Creates an instance of the <code>DNBuilder</code> implementation.
     * 
     * @return the configured implementation
     */
    public DNBuilder createDNBuilderInstance() {
        log.debug("Creating a new DNBuilder instance");
        String className = getImplementationClassName(this.configuration
                .getDNBuilderConfiguration());
        return (DNBuilder) this.createComponentInstance(className);
    }

    /**
     * Creates an instance of the <code>AAClient</code> implementation.
     * 
     * @return the configured implementation
     */
    public AAClient createAAClientInstance() {
        log.debug("Creating a new AAClient instance");
        String className = getImplementationClassName(this.configuration
                .getAAClientConfiguration());
        return (AAClient) this.createComponentInstance(className);
    }

    /**
     * Creates an instance of the <code>CAClient</code> implementation.
     * 
     * @return the configured implementation
     */
    public CAClient createCAClientInstance() {
        log.debug("Creating a new CAClient instance");
        String className = getImplementationClassName(this.configuration
                .getCAClientConfiguration());
        return (CAClient) this.createComponentInstance(className);
    }

    /**
     * Creates an instance of the <code>Auditor</code> implementation.
     * 
     * @return the configured implementation
     */
    public Auditor createAuditorInstance() {
        log.debug("Creating a new Auditor instance");
        String className = getImplementationClassName(this.configuration
                .getAuditorConfiguration());
        return (Auditor) this.createComponentInstance(className);
    }

    /**
     * Creates an instance of the <code>AuditAdmin</code> implementation.
     * 
     * @return the configured implementation
     */
    public AuditAdmin createAuditAdminInstance() {
        log.debug("Creating a new AuditAdmin instance");
        String className = getImplementationClassName(this.configuration
                .getAuditAdminConfiguration());
        return (AuditAdmin) this.createComponentInstance(className);
    }

    /**
     * Creates an instance of the <code>AccessControlList</code> implementation.
     * 
     * @return the configured implementation
     */
    public AccessControlList createAccessControlListInstance() {
        log.debug("Creating a new AccessControl instance");
        String className = getImplementationClassName(this.configuration
                .getAccessControlListConfiguration());
        return (AccessControlList) this.createComponentInstance(className);
    }

    /**
     * Creates an instance of the <code>AccessControlList</code> implementation
     * for administrator ACL.
     * 
     * @return the configured implementation
     */
    public AccessControlList createAdministratorAccessControlListInstance() {
        log.debug("Creating a new AdminAccessControl instance");
        String className = getImplementationClassName(this.configuration
                .getAdminAccessControlListConfiguration());
        return (AccessControlList) this.createComponentInstance(className);
    }

    /**
     * Creates an instance of the given class, implementing the
     * <code>PseudoServerComponent</code> interface.
     * 
     * @param className
     *            the name of the class as <code>String</code>
     * @return an instance of the class
     */
    protected PseudoServerComponent createComponentInstance(String className) {
        log.debug("Creating a new instance of the class {}", className);
        if (className.toLowerCase().trim().equals("none")) {
        	return null;
        }
        PseudoServerComponent builder = null;
        try {
            builder = (PseudoServerComponent) Class.forName(className)
                    .newInstance();
            builder.init(this.configuration);
        } catch (InstantiationException e) {
            log.error("Could not build a new instance of " + className, e);
        } catch (IllegalAccessException e) {
            log.error("Illegal access error for the class " + className, e);
        } catch (ClassNotFoundException e) {
            log.error("Could not find the implementation class " + className, e);
        } catch (PseudoServerException e) {
            log.error("Could not initialize the implementation class "
                    + className, e);
        }
        return builder;
    }

    /**
     * Gets the configured class name for the implementation.
     * 
     * @param iniSection
     *            section of the <code>PseudoServerConfiguration</code>
     * @return the name of the class as <code>String</code>
     */
    protected static String getImplementationClassName(Ini.Section iniSection) {
        return iniSection.get(PseudoServerConfiguration.CFG_IMPLEMENTATION_ID);
    }
}
