/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.caclient.impl;

import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.Principal;
import java.security.cert.X509Certificate;

import org.apache.http.HttpResponse;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.x509.X509CertificateStructure;
import org.bouncycastle.jce.provider.X509CertificateObject;
import org.glite.pseudo.common.pki.Certificate;
import org.glite.pseudo.server.caclient.CAClientException;
import org.glite.pseudo.server.caclient.CAResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.novosec.pkix.asn1.cmp.CertRepMessage;
import com.novosec.pkix.asn1.cmp.CertResponse;
import com.novosec.pkix.asn1.cmp.PKIMessage;

/**
 * CMPResponse is an implementation for RFC 4210 response.
 */
public class CMPResponse implements CAResponse {

    /** Logging */
    private static Logger log = LoggerFactory.getLogger(CMPResponse.class);

    /** Related PKIMessage */
    private PKIMessage pkiMessage;

    /**
     * Constructs a new <code>CMPResponse</code>.
     * 
     * @param caResponse
     *            the response object from the online CA
     * @throws CAClientException
     *             if something goes wrong
     */
    public CMPResponse(HttpResponse caResponse) throws CAClientException {
        if (caResponse == null) {
            throw new CAClientException("The response was empty!");
        }
        try {
            InputStream input = caResponse.getEntity().getContent();
            this.pkiMessage = PKIMessage.getInstance(new ASN1InputStream(input)
                    .readObject());
            // required for releasing the HttpClient's connection
            input.close();
        } catch (IOException e) {
            log.error("Error while creating PKIMessage object!", e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.caclient.CAResponse#getCertificate(java.security
     * .Principal)
     */
    public Certificate getCertificate(Principal subject)
            throws CAClientException {
        CertRepMessage certRepMessage = this.pkiMessage.getBody().getCp();
        // Get the first message in the chain
        CertResponse certResp = certRepMessage.getResponse(0);
        if (certResp == null) {
            log.error("No certificates found from the response!");
            return null;
        }
        X509CertificateStructure certSt = certResp.getCertifiedKeyPair()
                .getCertOrEncCert().getCertificate();
        X509CertificateObject certObject = null;
        try {
            // generate the certificate object
            certObject = new X509CertificateObject(certSt);
        } catch (Exception e) {
            log.error("Error while creating certObject: " + e);
            throw new CAClientException(e);
        }
        if (certObject.getSubjectDN().equals(subject)) {
            log.info("The certificate subject matched with the requested one.");
            X509Certificate cert = (X509Certificate) certObject;
            try {
                return new Certificate(cert);
            } catch (GeneralSecurityException e) {
                throw new CAClientException(e);
            }
        } else {
            log.warn("The certificate subject did NOT match with the requested one!");
            return null;
        }
    }

}
