/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.caclient.impl;

import java.io.IOException;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;
import org.glite.pseudo.common.pki.CertificateRequest;
import org.glite.pseudo.server.caclient.CAClientException;
import org.glite.pseudo.server.caclient.CAConnection;
import org.glite.pseudo.server.caclient.CARequest;
import org.glite.pseudo.server.caclient.CAResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * CMP-implementation for the <code>CAConnection</code>.
 */
public class CMPConnection implements CAConnection {

    /** Logging */
    private static Logger log = LoggerFactory.getLogger(CMPConnection.class);

    /** the CMPClient associated with this connection */
    private CMPClient cmpClient;

    /** The response container */
    private CMPResponse cmpResponse;

    /**
     * Constructs a new <code>CMPConnection</code>.
     * 
     * @param caClient
     *            the attached <code>CMPClient</code>
     */
    protected CMPConnection(CMPClient caClient) {
        this.cmpClient = caClient;
        this.cmpResponse = null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.caclient.CAConnection#createRequest(org.glite
     * .pseudo.common.pki.CertificateRequest)
     */
    public CARequest createRequest(CertificateRequest csr)
            throws CAClientException {
        CMPRequest cmpRequest = new CMPRequest(csr,
                cmpClient.getCmpProperties());
        return cmpRequest;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.caclient.CAConnection#sendRequest(org.glite.pseudo
     * .server.caclient.CARequest)
     */
    public void sendRequest(CARequest caRequest) throws CAClientException {
        byte[] requestBytes = caRequest.getDEREncoded();
        ByteArrayEntity entity = new ByteArrayEntity(requestBytes);
        HttpPost cmpServerPost = new HttpPost(this.cmpClient.getServerUrl());
        cmpServerPost.setEntity(entity);
        HttpClient httpClient = this.cmpClient.getHttpClient();
        HttpResponse postResponse = null;
        try {
            log.debug("Sending the request to the online CA url: {}",
                    this.cmpClient.getServerUrl());
            postResponse = httpClient.execute(cmpServerPost);
            log.debug("The response from the online CA: {}", postResponse);
        } catch (IOException e) {
            log.error("IO error while sending the request to the online CA.", e);
            throw new CAClientException(e);
        }
        this.cmpResponse = new CMPResponse(postResponse);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.server.caclient.CAConnection#getResponse()
     */
    public CAResponse getResponse() throws CAClientException {
        if (this.cmpResponse == null) {
            throw new CAClientException("Response object is null!");
        }
        return this.cmpResponse;
    }

}
