/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.caclient.impl;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.Properties;

import org.apache.http.client.HttpClient;
import org.glite.pseudo.common.http.HttpClientBuilder;
import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.caclient.CAClient;
import org.glite.pseudo.server.caclient.CAClientException;
import org.glite.pseudo.server.caclient.CAConnection;
import org.glite.pseudo.server.config.PseudoServerConfiguration;
import org.ini4j.Ini;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.novosec.pkix.asn1.cmp.CMPObjectIdentifiers;

/**
 * <p>
 * CMPClient is a RFC4210 compliant online CA client. This implementation uses a
 * HttpClient with clientAuth capabilities to send the requests and get the
 * responses.
 * </p>
 * 
 * <p>
 * Required configuration variables:
 * <ul>
 * <li><code>Endpoint</code> The URL for the online CA's CMP service
 * <li><code>CertificateFile</code> The certificate used for client
 * authentication
 * <li><code>KeyFile</code> The private key corresponding to the certificate
 * <li><code>TruststoreFile</code> Truststore for verifying the server
 * certificate
 * <li><code>TruststorePasswd</code>
 * <li><code>CADN</code> Online CA's subject DN
 * <li><code>SenderDN</code> RA's subject DN
 * <li><code>RecipientDN</code> Online CA's CMP service subject DN
 * <li><code>SenderKID</code> RA's key identifier
 * <li><code>SharedSecret</code> Shared secret between the RA and the online CA
 * </ul>
 * </p>
 * 
 * <p>
 * Optional variables:
 * <ul>
 * <li><code>KeyPasswd</code> Password used for decrypting the private key
 * <li><code>OwfAlgId</code> (default: 1.3.14.3.2.26)
 * <li><code>IterCount</code> (default: 1)
 * <li><code>MacAlgId</code> (default: 1.3.6.1.5.5.8.1.2)
 * <li><code>SaltString</code> (default: empty string)
 * <li><code>ProtectionAlgId</code> (default: 1.2.840.113533.7.66.13
 * (passwordBasedMac))
 * </ul>
 * </p>
 */
public class CMPClient extends AbstractSSLClient implements CAClient {

    /** Configuration identifier for the DN of the CA */
    public static final String CA_DN_IDENTIFIER = "CADN";

    /** Configuration identifier for the DN of the sender (this service) */
    public static final String SENDER_DN_IDENTIFIER = "SenderDN";

    /** Configuration identifier for the DN of the recipient */
    public static final String RECIPIENT_DN_IDENTIFIER = "RecipientDN";

    /** Configuration identifier for the key identifier */
    public static final String SENDER_KID_IDENTIFIER = "SenderKID";

    /** Configuration identifier for the shared secret */
    public static final String SHARED_SECRET_IDENTIFIER = "SharedSecret";

    /** Configuration identifier for the owf algorithm */
    public static final String OWF_ALG_ID_IDENTIFIER = "OwfAlgId";

    /** Configuration identifier for the iteration count */
    public static final String ITERATION_COUNT_IDENTIFIER = "IterCount";

    /** Configuration identifier for the mac algorithm */
    public static final String MAC_ALG_ID_IDENTIFIER = "MacAlgId";

    /** Configuration identifier for the salt string */
    public static final String SALT_STRING_IDENTIFIER = "SaltString";

    /** Configuration identifier for the protection algorithm */
    public static final String PROTECTION_ALG_ID_IDENTIFIER = "ProtectionAlgId";

    /** Default owf algorithm identifier */
    public static final String DEFAULT_OWF_ALGID = "1.3.14.3.2.26";

    /** Default iteration count */
    public static final String DEFAULT_ITERATION_COUNT = "1";

    /** Default mac algorithm identifier */
    public static final String DEFAULT_MAC_ALGID = "1.3.6.1.5.5.8.1.2";

    /** Default protection algorithm identifier */
    public static final String DEFAULT_PROTECTION_ALGID = CMPObjectIdentifiers.passwordBasedMac
            .toString();

    /** Logging */
    static private Logger log = LoggerFactory.getLogger(CMPClient.class);

    /** The HTTP client used for the connection */
    private HttpClient httpClient;

    /** Configuration properties for the connection */
    private Properties cmpProperties;

    /**
     * Constructs a new <code>CMPClient</code>.
     */
    public CMPClient() {
        // no op
    }

    /**
     * Sets the CMP configuration properties for the connection.
     * 
     * @param cfgSection
     *            configuration section from the server configuration
     * @throws CAClientException
     *             if mandatory variables are missing
     */
    protected void setCmpProperties(Ini.Section cfgSection)
            throws CAClientException {
        this.cmpProperties = new Properties();
        this.generateCmpProperties(cfgSection);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.slcs.caclient.CAClient#getConnection()
     */
    public CAConnection getConnection() throws CAClientException {
        return new CMPConnection(this);
    }

    private void generateCmpProperties(Ini.Section cfgSection)
            throws CAClientException {
        addCmpConfigurationVariable(cfgSection, CA_DN_IDENTIFIER, null);
        addCmpConfigurationVariable(cfgSection, SENDER_DN_IDENTIFIER, null);
        addCmpConfigurationVariable(cfgSection, RECIPIENT_DN_IDENTIFIER, null);
        addCmpConfigurationVariable(cfgSection, SENDER_KID_IDENTIFIER, null);
        addCmpConfigurationVariable(cfgSection, SHARED_SECRET_IDENTIFIER, null);
        addCmpConfigurationVariable(cfgSection, OWF_ALG_ID_IDENTIFIER,
                DEFAULT_OWF_ALGID);
        addCmpConfigurationVariable(cfgSection, ITERATION_COUNT_IDENTIFIER,
                DEFAULT_ITERATION_COUNT);
        addCmpConfigurationVariable(cfgSection, MAC_ALG_ID_IDENTIFIER,
                DEFAULT_MAC_ALGID);
        addCmpConfigurationVariable(cfgSection, SALT_STRING_IDENTIFIER, "");
        addCmpConfigurationVariable(cfgSection, PROTECTION_ALG_ID_IDENTIFIER,
                DEFAULT_PROTECTION_ALGID);
    }

    private void addCmpConfigurationVariable(Ini.Section cfgSection,
            String variable, String defaultValue) throws CAClientException {
        String str = cfgSection.get(variable);
        if (str == null || str.equals("")) {
            if (defaultValue == null) {
                throw new CAClientException(variable
                        + " is a required variable!");
            } else {
                log.info("CAClient." + variable + "='" + defaultValue
                        + "' (was null, using default)");
                this.cmpProperties.setProperty(variable, defaultValue);
            }
        } else {
            this.cmpProperties.setProperty(variable, str);
            log.info(variable + "=" + str);
        }
    }

    private HttpClient createHttpClient(String keyFile, String keyPasswd,
            String certFile, String truststorePath, String truststorePasswd,
            String crlCheck) throws CAClientException {
        HttpClientBuilder httpBuilder = new HttpClientBuilder(this.endpoint,
                certFile, keyFile, keyPasswd);
        HttpClient httpClient = null;
        try {
            httpClient = httpBuilder.buildClient();
        } catch (Exception e) {
            log.error("Could not build the httpClient!", e);
            throw new CAClientException("The HTTP client could not be built!",
                    e);
        }
        return httpClient;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.server.caclient.impl.AbstractSSLClient#shutdown()
     */
    public void shutdown() {
        super.shutdown();
        this.httpClient.getConnectionManager().shutdown();
        this.httpClient = null;
    }

    /**
     * @return the underlying HttpClient
     */
    protected HttpClient getHttpClient() {
        return this.httpClient;
    }

    /**
     * @return the CMP configuration variables
     */
    protected Properties getCmpProperties() {
        return this.cmpProperties;
    }

    /**
     * @return the online CA's CMP service url
     */
    protected String getServerUrl() {
        return this.endpoint;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.PseudoServerComponent#init(org.glite.pseudo.server
     * .config.PseudoServerConfiguration)
     */
    public void init(PseudoServerConfiguration configuration)
            throws PseudoServerException {
        log.debug("Initializing CMPClient");
        Ini.Section cfgSection = configuration.getCAClientConfiguration();
        super.init(cfgSection);
        try {
            this.httpClient = createHttpClient(keyFile, keyPasswd, certFile,
                    truststoreFile, truststorePasswd, crlCheck);
            this.setCmpProperties(cfgSection);
        } catch (CAClientException e) {
            log.error("Could not initialize a CMPClient", e);
            throw new PseudoServerException("Could not initialize a CMPClient",
                    e);
        }
    }

}