/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.auditor.impl;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.Date;
import java.util.Map;

import org.glite.pseudo.server.Constants;
import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.auditor.AuditEventEntry;
import org.glite.pseudo.server.auditor.AuditPersonEntry;
import org.glite.pseudo.server.auditor.Auditor;
import org.glite.pseudo.server.auditor.AuditorException;
import org.glite.pseudo.server.auditor.event.AuditEvent;
import org.glite.pseudo.server.config.PseudoServerConfiguration;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A Hibernate-based implementation for the Auditor-interface.
 */
public class HibernateAuditor extends AbstractHibernateAuditor implements
        Auditor {

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(HibernateAuditor.class);

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.auditor.Auditor#logEvent(org.glite.pseudo.server
     * .auditor.event.AuditEvent)
     */
    public void logEvent(AuditEvent event) throws AuditorException {
        Map<String, String> attributesMap = event.getAttributesMap();
        String subjectDN = (String) attributesMap
                .get(Constants.USER_SUBJECT_DN_ATTR_IDENTIFIER);
        String remoteAddress = (String) attributesMap
                .get(Constants.USER_REMOTE_ADDRESS_ATTR_IDENTIFIER);

        Session session = this.getCurrentSession();
        Transaction tx = null;
        try {
            tx = session.beginTransaction();

            // check whether the user already exist
            Query hqlQuery = session.createQuery("from " + this.personImpl
                    + " as person where person.identifier like :subjectDN");
            hqlQuery.setString("subjectDN", subjectDN);
            AuditPersonEntry person = (AuditPersonEntry) hqlQuery
                    .uniqueResult();
            // if not, construct a new one
            if (person == null) {
                log.info("New user: {}", subjectDN);
                try {
                    person = this.makePersonEntry(subjectDN);
                } catch (Exception e) {
                    log.error("Error while constructing a new "
                            + this.personImpl, e);
                    throw e;
                }
                session.save(person);
                log.debug("New user saved.");
            } else {
                log.debug("Existing user: {}", subjectDN);
            }

            AuditEventEntry eventEntry = null;
            try {
                eventEntry = this.makeEventEntry(event, remoteAddress, person);
            } catch (Exception e) {
                log.error("Error while constructing a new {}", this.eventImpl,
                        e);
                throw e;
            }
            // bind the event and the user
            person.addAuditEvent(eventEntry);

            // store the user and the event
            session.save(eventEntry);
            session.save(person);
            tx.commit();
        } catch (Exception e) {
            tx.rollback();
            throw new AuditorException(e);
        } finally {
            closeSession(session);
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.auditor.impl.AbstractHibernateAuditor#init(org
     * .glite.pseudo.server.config.PseudoServerConfiguration)
     */
    public void init(PseudoServerConfiguration configuration)
            throws PseudoServerException {
        log.debug("Initializing HibernateAuditor..");
        super.init(configuration);
        log.info("HibernateAuditor successfully initialized.");
    }

    // PRIVATE METHODS

    private AuditEventEntry makeEventEntry(AuditEvent event,
            String remoteAddress, AuditPersonEntry person) throws Exception {
        // Construct a new object implementing AuditEventEntry
        Constructor<?> constructor = Class.forName(this.eventImpl)
                .getConstructor(new Class[0]);
        Object obj = constructor.newInstance(new Object[0]);
        // Temporary variables for invoking methods
        Class<?>[] argClass = new Class[1];
        Object[] argObject = new Object[1];
        // setEventDate(Calendar.getInstance().getTime())
        Array.set(argClass, 0, Date.class);
        Array.set(argObject, 0, Calendar.getInstance().getTime());
        obj.getClass().getMethod("setEventDate", argClass)
                .invoke(obj, argObject);
        // setEventLevel(event.getLevel())
        Array.set(argClass, 0, int.class);
        Array.set(argObject, 0, new Integer(event.getLevel()));
        obj.getClass().getMethod("setEventLevel", argClass)
                .invoke(obj, argObject);
        // setEventMessage(event.getMessage())
        Array.set(argClass, 0, String.class);
        Array.set(argObject, 0, event.getMessage());
        obj.getClass().getMethod("setEventMessage", argClass)
                .invoke(obj, argObject);
        // setEventType(event.getType())
        Array.set(argClass, 0, int.class);
        Array.set(argObject, 0, new Integer(event.getType()));
        obj.getClass().getMethod("setEventType", argClass)
                .invoke(obj, argObject);
        // setRemoteAddress(remoteAddress)
        Array.set(argClass, 0, String.class);
        Array.set(argObject, 0, remoteAddress);
        obj.getClass().getMethod("setRemoteAddress", argClass)
                .invoke(obj, argObject);
        // setEventAuthor(person)
        Array.set(argClass, 0, AuditPersonEntry.class);
        Array.set(argObject, 0, person);
        obj.getClass().getMethod("setEventAuthor", argClass)
                .invoke(obj, argObject);
        // all done
        return (AuditEventEntry) obj;
    }

    private AuditPersonEntry makePersonEntry(String subjectDN) throws Exception {
        Class<?>[] classes = { subjectDN.getClass() };
        Object[] initargs = { subjectDN };
        // no argument constructor
        Constructor<?> constructor = Class.forName(this.personImpl)
                .getConstructor(new Class[0]);
        Object obj = constructor.newInstance(new Object[0]);
        // setIdentifier(subjectDN)
        Method method = obj.getClass().getMethod("setIdentifier", classes);
        method.invoke(obj, initargs);
        // all done
        return (AuditPersonEntry) obj;
    }

}
