/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.auditor.impl;

import java.util.List;

import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.auditor.AuditAdmin;
import org.glite.pseudo.server.auditor.AuditEventEntry;
import org.glite.pseudo.server.auditor.AuditPersonEntry;
import org.glite.pseudo.server.config.PseudoServerConfiguration;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A Hibernate-based implementation for the AuditAdmin-interface.
 */
public class HibernateAuditAdmin extends AbstractHibernateAuditor implements
        AuditAdmin {

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(HibernateAuditAdmin.class);

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.audit.admin.AuditAdmin#getEvent(java.lang.String)
     */
    public AuditEventEntry getEvent(String eventId) {
        log.debug("Retrieving event details for the eventId={}", eventId);
        Session session = this.getCurrentSession();
        Transaction tx = null;
        AuditEventEntry result = null;
        try {
            tx = session.beginTransaction();
            result = queryEvent(eventId, session);
            tx.commit();
        } catch (Exception e) {
            log.error("Exception caught while querying the database", e);
            tx.rollback();
        } finally {
            closeSession(session);
        }
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.audit.admin.AuditAdmin#getEvents(java.lang.String)
     */
    public AuditEventEntry[] getEvents(String personId) {
        return this.getEvents(personId, null);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.audit.admin.AuditAdmin#getEvents(java.lang.String,
     * java.lang.String)
     */
    public AuditEventEntry[] getEvents(String personId, String sortedBy) {
        log.debug("Retrieving all the events for the personId={}", personId);
        Session session = this.getCurrentSession();
        Transaction tx = null;
        AuditEventEntry[] result = null;
        try {
            tx = session.beginTransaction();
            AuditPersonEntry person = queryPerson(personId, session);
            result = person.getAuditEvents();
            if (sortedBy == null) {
                // TODO
            } else {
                // TODO
            }
            tx.commit();
        } catch (Exception e) {
            log.error("Exception caught while querying the database", e);
            tx.rollback();
        } finally {
            closeSession(session);
        }
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.audit.admin.AuditAdmin#getPerson(java.lang.String)
     */
    public AuditPersonEntry getPerson(String personId) {
        log.debug("Retrieving user details for the personId={}", personId);
        Session session = this.getCurrentSession();
        Transaction tx = null;
        AuditPersonEntry result = null;
        try {
            tx = session.beginTransaction();
            result = queryPerson(personId, session);
            tx.commit();
        } catch (Exception e) {
            log.error("Exception caught while querying the database", e);
            tx.rollback();
        } finally {
            closeSession(session);
        }
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.audit.admin.AuditAdmin#getPersons()
     */
    public AuditPersonEntry[] getPersons() {
        log.debug("Retrieving all the users from the database");
        Session session = this.getCurrentSession();
        Transaction tx = null;
        List<AuditPersonEntry> list = null;
        try {
            tx = session.beginTransaction();
            Query query = session.createQuery("from " + this.personImpl);
            list = (List<AuditPersonEntry>) query.list();
            tx.commit();
        } catch (Exception e) {
            log.error("Exception caught while querying the database", e);
            tx.rollback();
        } finally {
            closeSession(session);
        }
        AuditPersonEntry[] result = new AuditPersonEntry[list.toArray().length];
        list.toArray(result);
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.audit.admin.AuditAdmin#getPseudoEvents(java.lang.String)
     */
    public AuditEventEntry[] getPseudoEvents(String pseudoDN) {
        return getPseudoEvents(pseudoDN, null);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.audit.admin.AuditAdmin#getPseudoEvents(java.lang.String,
     * java.lang.String)
     */
    public AuditEventEntry[] getPseudoEvents(String pseudoDN, String sortedBy) {
        return getEventsByMessage(pseudoDN, sortedBy);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.auditor.impl.AbstractHibernateAuditor#init(org
     * .glite.pseudo.server.config.PseudoServerConfiguration)
     */
    public void init(PseudoServerConfiguration configuration)
            throws PseudoServerException {
        log.debug("Initializing HibernateAuditAdmin..");
        super.init(configuration);
        log.info("HibernateAuditAdmin successfully initialized.");
    }

    // PRIVATE METHODS

    /**
     * <p>
     * Queries the specific audit event author with the specific Hibernate
     * session.
     * </p>
     * 
     * @param personId
     *            The identifier for the author
     * @param session
     *            The Hibernate session
     * @return The author as AuditPersonEntry
     */
    private AuditPersonEntry queryPerson(String personId, Session session) {
        Query query = session.createQuery("from " + this.personImpl
                + " as person where person.id like :personId");
        query.setString("personId", personId);
        return (AuditPersonEntry) query.uniqueResult();
    }

    /**
     * <p>
     * Queries the specific audit event with the specific Hibernate session.
     * </p>
     * 
     * @param eventId
     *            The identifier for the event
     * @param session
     *            The Hibernate session
     * @return The event as AuditEventEntry
     */
    private AuditEventEntry queryEvent(String eventId, Session session) {
        Query query = session.createQuery("from " + this.eventImpl
                + " as event where event.id like :eventId");
        query.setString("eventId", eventId);
        return (AuditEventEntry) query.uniqueResult();
    }

    /**
     * <p>
     * Queries the database with the specific event message.
     * </p>
     * 
     * @param message
     *            The message for the event
     * @param sortedBy
     *            The column used for sorting
     * @return All the matching events
     */
    private AuditEventEntry[] getEventsByMessage(String message, String sortedBy) {
        log.debug("Retrieving all the events for message={}", message);
        Session session = this.getCurrentSession();
        Transaction tx = null;
        List<AuditEventEntry> list = null;
        try {
            tx = session.beginTransaction();
            Query query = null;
            if (sortedBy == null) {
                query = session.createQuery("from " + this.eventImpl
                        + " as event where event.eventMessage like :message");
                query.setString("message", message);
            } else {
                query = session.createQuery("from " + this.eventImpl
                        + " as event where event.eventMessage like :message "
                        + " order by event." + sortedBy);
            }
            query.setString("message", message);
            list = (List<AuditEventEntry>) query.list();
        } catch (Exception e) {
            log.error("Exception caught while querying the database", e);
            tx.rollback();
        }
        AuditEventEntry[] result = new AuditEventEntry[list.toArray().length];
        list.toArray(result);
        return result;
    }

}
