/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.auditor.impl;

import java.util.HashSet;
import java.util.Set;

import org.glite.pseudo.server.auditor.AuditEventEntry;
import org.glite.pseudo.server.auditor.AuditPersonEntry;

/**
 * <p>
 * Simple implementation for AuditPersonEntry interface.
 * </p>
 */
public class AuditPersonEntryImpl implements AuditPersonEntry {

    /** The unique Hibernate identifier for the person */
    private Long id;

    /** The unique identifier (like subject DN) for the person */
    private String identifier;

    /** The set of events related to the person */
    private Set<AuditEventEntry> events = new HashSet<AuditEventEntry>();

    /**
     * <p>
     * Constructs a new AuditPersonEntryImpl.
     * </p>
     */
    public AuditPersonEntryImpl() {
        // no op
    }

    /**
     * <p>
     * Constructs a new AuditPersonEntryImpl with a specific identifier. This
     * type of constructor is required by
     * org.glite.pseudo.audit.impl.HibernateAuditor implementation.
     * </p>
     * 
     * @see org.glite.pseudo.audit.impl.HibernateAuditor
     */
    public AuditPersonEntryImpl(String identifier) {
        this.setIdentifier(identifier);
    }

    // INTERFACE METHODS

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.audit.AuditPersonEntry#getAuditEvents()
     */
    public AuditEventEntry[] getAuditEvents() {
        AuditEventEntry[] result = new AuditEventEntry[this.events.size()];
        this.events.toArray(result);
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.audit.AuditPersonEntry#getIdentifier()
     */
    public String getIdentifier() {
        return this.identifier;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.audit.AuditPersonEntry#addAuditEvent(org.glite.pseudo
     * .audit.AuditEventEntry)
     */
    public void addAuditEvent(AuditEventEntry event) {
        Set<AuditEventEntry> eventSet = this.getEvents();
        eventSet.add(event);
        this.setEvents(eventSet);
    }

    // METHODS FOR HIBERNATE

    /**
     * <p>
     * Gets the Hibernate identifier.
     * </p>
     * 
     * @return the id
     */
    public Long getId() {
        return this.id;
    }

    /**
     * <p>
     * Sets the Hibernate identifier.
     * </p>
     * 
     * @param id
     *            the id to set
     */
    public void setId(Long id) {
        this.id = id;
    }

    /**
     * <p>
     * Gets the events for the person.
     * </p>
     * 
     * @return the events
     */
    public Set<AuditEventEntry> getEvents() {
        return this.events;
    }

    /**
     * <p>
     * Sets the events for the person.
     * </p>
     * 
     * @param events
     *            the events to set
     */
    public void setEvents(Set<AuditEventEntry> events) {
        this.events = events;
    }

    /**
     * <p>
     * Sets the identifier for the person.
     * </p>
     * 
     * @param identifier
     *            the identifier to set
     */
    public void setIdentifier(String identifier) {
        this.identifier = identifier;
    }

    /**
     * <p>
     * Test whether an other instance is equal with this.
     * </p>
     * 
     * @param other
     *            the other instance
     * @return true if equal, false otherwise
     */
    public boolean equals(AuditPersonEntryImpl other) {
        if (this == other)
            return true;
        if (!(other instanceof AuditPersonEntryImpl))
            return false;
        if (!other.getIdentifier().equals(this.getIdentifier()))
            return false;
        return true;
    }
}
