/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.auditor.impl;

import java.io.File;
import java.io.FileInputStream;

import org.glite.pseudo.common.util.Util;
import org.glite.pseudo.server.PseudoServerComponent;
import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.config.PseudoServerConfiguration;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.ini4j.Ini;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Abstract class to be used for the auditors using <code>Hibernate</code>
 * library.
 */
public abstract class AbstractHibernateAuditor implements PseudoServerComponent {

    /** Logging */
    private static final Logger log = LoggerFactory
            .getLogger(AbstractHibernateAuditor.class);

    /** Configuration variable for the Hibernate configuration file. */
    protected final static String HIBERNATE_CFG_IDENTIFIER = "HibernateConfiguration";

    /** Configuration variable for the Hibernate mappings configuration file. */
    protected final static String HIBERNATE_MAPPINGS_IDENTIFIER = "HibernateMappings";

    /**
     * Configuration variable for the implementation class for the Person
     * interface.
     */
    protected final static String HIBERNATE_PERSON_IMPLEMENTATION = "PersonImplementation";

    /**
     * Configuration variable for the implementation class for the Event
     * interface.
     */
    protected final static String HIBERNATE_EVENT_IMPLEMENTATION = "EventImplementation";

    /** The name of the class implementing the Person interface */
    protected String personImpl;

    /** The name of the class implementing the Event interface */
    protected String eventImpl;

    /** Hibernate session factory */
    protected SessionFactory sessionFactory = null;

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.PseudoServerComponent#init(org.glite.pseudo.server
     * .config.PseudoServerConfiguration)
     */
    public void init(PseudoServerConfiguration configuration)
            throws PseudoServerException {
        Ini.Section cfgSection = configuration.getAuditorConfiguration();
        String hbrConfig = Util.safeTrimOrNullString(cfgSection
                .get(HIBERNATE_CFG_IDENTIFIER));
        log.debug("Hibernate configuration file = {}", hbrConfig);
        if (hbrConfig == null) {
            throw new PseudoServerException(
                    "Hibernate configuration file cannot be empty!");
        }
        String hbrMappings = Util.safeTrimOrNullString(cfgSection
                .get(HIBERNATE_MAPPINGS_IDENTIFIER));
        log.debug("Hibernate mappings configuration file = {}", hbrMappings);
        if (hbrMappings == null) {
            throw new PseudoServerException(
                    "Hibernate mappings configuration file cannot be empty!");
        }
        this.personImpl = Util.safeTrimOrNullString(cfgSection
                .get(HIBERNATE_PERSON_IMPLEMENTATION));
        log.debug("Person implementation class = {}", this.personImpl);
        if (this.personImpl == null) {
            throw new PseudoServerException(
                    "Person implementation class cannot be empty!");
        }
        this.eventImpl = Util.safeTrimOrNullString(cfgSection
                .get(HIBERNATE_EVENT_IMPLEMENTATION));
        log.debug("Event implementation class = {}", this.eventImpl);
        if (this.eventImpl == null) {
            throw new PseudoServerException(
                    "Event implementation class cannot be empty!");
        }

        try {
            FileInputStream fileInput = new FileInputStream(hbrMappings);
            File cfgFile = new File(hbrConfig);
            Configuration cfg = new Configuration().configure(cfgFile);
            cfg = cfg.addInputStream(fileInput);
            this.sessionFactory = cfg.buildSessionFactory();
        } catch (Throwable ex) {
            log.error("Initial SessionFactory creation failed." + ex);
            throw new PseudoServerException(ex);
        }
        log.info("HibernateAuditor successfully initialized.");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.server.PseudoServerComponent#shutdown()
     */
    public void shutdown() {
        this.sessionFactory.close();
    }

    /**
     * Returns a current Session from Hibernate session factory.
     * 
     * @return session the current <code>Session</code> from the
     *         <code>SessionFactory</code>
     */
    protected Session getCurrentSession() {
        return this.sessionFactory.getCurrentSession();
    }

    /**
     * Closes the given session, if it's open.
     * 
     * @param session
     *            the <code>Session</code> to be closed
     */
    protected static void closeSession(Session session) {
        if (session.isOpen()) {
            session.close();
        }
    }

}
