/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.attribute;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.FileConfiguration;
import org.apache.commons.configuration.XMLConfiguration;
import org.glite.pseudo.common.util.Util;
import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.config.PseudoServerConfiguration;
import org.ini4j.Ini;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Helper class for the AttributeDefintions.
 * 
 * Imported from org.glite.slcs.attribute.AttributeDefinitionsImpl
 * 
 * @author Valery Tschopp <tschopp@switch.ch>
 */
public class AttributeDefinitionsImpl implements AttributeDefinitions {

    public static final String CONFIG_FILE_IDENTIFIER = "ConfigFile";

    /** Logging */
    static private Logger log = LoggerFactory
            .getLogger(AttributeDefinitionsImpl.class);

    /** Map of name - {@link AttributeDefinition}s */
    private Map<String, AttributeDefinition> attributeDefinitions_ = null;

    /** List of required attribute names */
    private List<String> requiredAttributeNames_ = null;

    /** List of all defined attribute header names */
    private Map<String, String> headerNameMapping_ = null;

    /** The file based configuration */
    private FileConfiguration configuration_ = null;

    public AttributeDefinitionsImpl() {
        // no op
    }

    /**
     * Parses the attribute definitions XML file and creates the list of
     * attributeDefinitions.
     * 
     * @throws AttributeDefinitionException
     */
    private void parseAttributeDefinitions()
            throws AttributeDefinitionException {
        // create the map and lists
        attributeDefinitions_ = new HashMap<String, AttributeDefinition>();
        headerNameMapping_ = new HashMap<String, String>();
        requiredAttributeNames_ = new ArrayList<String>();

        // populate the map and lists
        List<String> attributeDefinitionNames = getList("AttributeDefinition[@name]");
        int nAttributeDefinitions = attributeDefinitionNames.size();
        for (int i = 0; i < nAttributeDefinitions; i++) {
            String name = (String) attributeDefinitionNames.get(i);
            String prefix = "AttributeDefinition(" + i + ")";
            // get HTTP header name
            String header = getString(prefix + "[@header]");
            // get the display name
            String displayName = getString(prefix + "[@displayName]");
            // required is optional (default: false)
            String required = getString(prefix + "[@required]", false);
            // casesensitive is optional (default: true)
            String caseSensitive = getString(prefix + "[@caseSensitive]", false);
            // add the attribute header - names in the mapping table
            headerNameMapping_.put(header, name);
            // create a new attribute definition
            AttributeDefinition attributeDef = new AttributeDefinition(name,
                    header, displayName);
            // required default: false
            if (required != null && required.equalsIgnoreCase("true")) {
                requiredAttributeNames_.add(name);
                attributeDef.setRequired(true);
            }
            // caseSensitive default: true
            if (caseSensitive != null
                    && caseSensitive.equalsIgnoreCase("false")) {
                attributeDef.setCaseSensitive(false);
            }
            // add in the attribute's definitions map
            attributeDefinitions_.put(name, attributeDef);

        }
        log.info("AttributeDefinitions=" + attributeDefinitions_);
        log.info("HeaderNameMapping=" + headerNameMapping_);
        log.info("RequiredAttributeNames=" + requiredAttributeNames_);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.slcs.attribute.AttributeDefinitions#createAttribute(java.lang
     * .String, java.lang.String)
     */
    public Attribute createAttribute(String name, String value) {
        Attribute attribute = new Attribute(name, value);
        AttributeDefinition attributeDefinition = (AttributeDefinition) attributeDefinitions_
                .get(name);
        if (attributeDefinition != null) {
            attribute.setDisplayName(attributeDefinition.getDisplayName());
            attribute.setCaseSensitive(attributeDefinition.isCaseSensitive());
            attribute.setRequired(attributeDefinition.isRequired());
        }
        return attribute;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.slcs.attribute.AttributeDefinitions#getUserAttributes(javax
     * .servlet.http.HttpServletRequest)
     */
    public List<Attribute> getUserAttributes(HttpServletRequest request) {
        // list of user attributes
        List<Attribute> attributes = new ArrayList<Attribute>();
        @SuppressWarnings("unchecked")
        Enumeration<String> headers = request.getHeaderNames();
        while (headers.hasMoreElements()) {
            String headerName = (String) headers.nextElement();
            if (headerNameMapping_.containsKey(headerName)) {
                String headerValue = request.getHeader(headerName);
                // add only not null and not empty attributes
                if (headerValue != null && !headerValue.equals("")) {
                    String decodedValue = "";
                    try {
                        decodedValue = Util.convertUTF8ToUnicode(headerValue);
                    } catch (UnsupportedEncodingException e) {
                        log.error("Error while encoding {} to Unicode",
                                headerValue);
                    }
                    // multi-value attributes are stored as multiple attributes
                    String[] attrValues = decodedValue.split(";");
                    for (int i = 0; i < attrValues.length; i++) {
                        String attrName = getAttributeName(headerName);
                        String attrValue = attrValues[i];
                        attrValue = attrValue.trim();
                        Attribute attribute = new Attribute(attrName, attrValue);
                        AttributeDefinition attributeDefinition = (AttributeDefinition) attributeDefinitions_
                                .get(attrName);
                        if (attributeDefinition != null) {
                            attribute.setDisplayName(attributeDefinition
                                    .getDisplayName());
                            attribute.setCaseSensitive(attributeDefinition
                                    .isCaseSensitive());
                            attribute.setRequired(attributeDefinition
                                    .isRequired());
                        }
                        attributes.add(attribute);
                    }
                }

            }
        }

        if (log.isDebugEnabled()) {
            log.debug("attributes=" + attributes);
        }
        return attributes;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.slcs.attribute.AttributeDefinitions#getAttributeDefinitionsList
     * ()
     */
    public List<AttributeDefinition> getAttributeDefinitions() {
        Collection<AttributeDefinition> definitions = attributeDefinitions_
                .values();
        List<AttributeDefinition> attributeDefinitions = new ArrayList<AttributeDefinition>(
                definitions);
        Collections.sort(attributeDefinitions,
                new AttributeDefinitionComparator());
        return attributeDefinitions;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.slcs.attribute.AttributeDefinitions#getDisplayName(org.glite
     * .slcs.attribute.Attribute)
     */
    public String getAttributeDisplayName(Attribute attribute) {
        String attrName = attribute.getName();
        if (attributeDefinitions_.containsKey(attrName)) {
            AttributeDefinition attrDef = (AttributeDefinition) attributeDefinitions_
                    .get(attrName);
            return attrDef.getDisplayName();
        }
        return null;
    }

    /**
     * Returns the attribute name for the given HTTP header.
     * 
     * @param header
     *            The HTTP header name.
     * @return The attribute name or <code>null</code> if the header is not
     *         mapped.
     */
    private String getAttributeName(String header) {
        if (headerNameMapping_.containsKey(header)) {
            String name = (String) headerNameMapping_.get(header);
            return name;
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.slcs.attribute.AttributeDefinitions#isRequired(org.glite.slcs
     * .attribute.Attribute)
     */
    public boolean isAttributeRequired(Attribute attribute) {
        String attrName = attribute.getName();
        if (attributeDefinitions_.containsKey(attrName)) {
            AttributeDefinition attrDef = (AttributeDefinition) attributeDefinitions_
                    .get(attrName);
            return attrDef.isRequired();
        } else {
            return AttributeDefinition.DEFAULT_REQUIRED;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.slcs.attribute.AttributeDefinitions#isAttributeCaseSensitive
     * (org.glite.slcs.attribute.Attribute)
     */
    public boolean isAttributeCaseSensitive(Attribute attribute) {
        String attrName = attribute.getName();
        if (attributeDefinitions_.containsKey(attrName)) {
            AttributeDefinition attrDef = (AttributeDefinition) attributeDefinitions_
                    .get(attrName);
            return attrDef.isCaseSensitive();
        } else {
            return AttributeDefinition.DEFAULT_CASE_SENSITIVE;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.slcs.attribute.AttributeDefinitions#setDisplayNames(java.util
     * .List)
     */
    public void setDisplayNames(List<Attribute> attributes) {
        Iterator<Attribute> iter = attributes.iterator();
        while (iter.hasNext()) {
            Attribute attr = (Attribute) iter.next();
            String displayName = getAttributeDisplayName(attr);
            attr.setDisplayName(displayName);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.slcs.attribute.AttributeDefinitions#getRequiredAttributeNames()
     */
    public List<String> getRequiredAttributeNames() {
        return requiredAttributeNames_;
    }

    /**
     * Creates a XMLConfiguration loaded with the given file.
     * 
     * @param filename
     *            The file name to load the XML configuration from.
     * @return The new FileConfiguration object
     * @throws SLCSConfigurationException
     *             If a configuration error occurs while loading the XML file.
     */
    protected FileConfiguration loadConfiguration(String filename)
            throws AttributeDefinitionException {
        XMLConfiguration config = null;
        try {
            log.info("XMLConfiguration file=" + filename);
            config = new XMLConfiguration(filename);
            if (log.isDebugEnabled()) {
                File configFile = config.getFile();
                log.debug("XMLConfiguration file="
                        + configFile.getAbsolutePath());
            }
        } catch (ConfigurationException e) {
            log.error("Failed to create XMLConfiguration: " + filename, e);
            throw new AttributeDefinitionException(
                    "Failed to create XMLConfiguration: " + filename, e);
        }
        return config;
    }

    /**
     * Sets the FileConfiguration and checks for validity.
     * 
     * @param configuration
     *            The FileConfiguration to set.
     * @throws AttributeDefinitionException
     */
    protected void setFileConfiguration(FileConfiguration configuration)
            throws AttributeDefinitionException {
        this.configuration_ = configuration;
    }

    /**
     * @param name
     *            The configuration key.
     * @return The associated List. Empty if the name is not in configuration.
     */
    public List<String> getList(String name) {
        @SuppressWarnings("unchecked")
        List<String> list = configuration_.getList(name);
        return list;
    }

    /**
     * Returns the value of the key and throw exception if the key is not
     * defined.
     * 
     * @param name
     *            The key of the value to get
     * @return The value of for this key
     * @throws AttributeDefinitionException
     *             if the key is missing from configuration or empty
     */
    public String getString(String name) throws AttributeDefinitionException {
        return getString(name, true);
    }

    /**
     * Returns the value of the key and throw exception if the key is not
     * defined only if throwException is <code>true</code>.
     * 
     * @param name
     *            The key name of the value to read.
     * @param throwException
     *            Throw an exception if the key is not found or not.
     * @return The value or <code>null</code> if the key is not found or the
     *         value empty.
     * @throws AttributeDefinitionException
     */
    public String getString(String name, boolean throwException)
            throws AttributeDefinitionException {
        String value = configuration_.getString(name);
        if (value == null || value.equals("")) {
            value = null;
            if (throwException) {
                throw new AttributeDefinitionException(name
                        + " is null or empty: " + getFilename());
            }
        }
        return value;
    }

    /**
     * @return The XML configuration absolute filename.
     */
    protected String getFilename() {
        return this.configuration_.getFile().getAbsolutePath();
    }

    public void init(PseudoServerConfiguration configuration)
            throws PseudoServerException {
        Ini.Section cfgSection = configuration
                .getAttributeDefinitionsConfiguration();
        String filename = Util.safeTrimOrNullString(cfgSection
                .get(CONFIG_FILE_IDENTIFIER));
        if (filename == null) {
            throw new PseudoServerException(
                    "Attribute definition configuration file cannot be empty!");
        }
        log.debug("filename: {}", filename);
        try {
            FileConfiguration fileConfiguration = loadConfiguration(filename);
            // setFileConfiguration call checkConfiguration...
            setFileConfiguration(fileConfiguration);
            parseAttributeDefinitions();
        } catch (AttributeDefinitionException e) {
            throw new PseudoServerException(
                    "Could not parse the attribute definition configuration!",
                    e);
        }
    }

    public void shutdown() {
        // TODO Auto-generated method stub

    }
}
