/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.acl.impl;

import java.io.File;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.servlet.FilterConfig;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.FileConfiguration;
import org.apache.commons.configuration.XMLConfiguration;
import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.acl.AccessControlList;
import org.glite.pseudo.server.acl.AccessControlRule;
import org.glite.pseudo.server.attribute.Attribute;
import org.glite.pseudo.server.attribute.AttributeDefinitions;
import org.glite.pseudo.server.config.PseudoServerConfiguration;
import org.ini4j.Ini;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

/**
 * XMLFileAccessControlList implements a XML file based ACL. This implementation
 * use a FileConfigurationMonitor to track the file modications and reload it on
 * changes.
 * 
 * @author Valery Tschopp <tschopp@switch.ch>
 * 
 *         Imported from org.glite.slcs.acl.impl.XMLFileAccessControlList
 */
public class XMLFileAccessControlList implements AccessControlList,
        ApplicationContextAware {

    /** Name of the ACL file parameter in the {@link FilterConfig} */
    private static String ACLFILE_CONFIG_PARAM = "ACLFile";

    /** Logging */
    private static Logger log = LoggerFactory
            .getLogger(XMLFileAccessControlList.class);

    /** XML file based authorization */
    private XMLConfiguration aclXMLConfiguration_ = null;

    /** List of Access Control Rules */
    private List<AccessControlRule> accessControlRules_ = null;

    /** Spring application context */
    private ApplicationContext context;

    /**
     * Constructor called by the factory.
     */
    public XMLFileAccessControlList() {
        super();
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.slcs.acl.AccessControlList#init(javax.servlet.FilterConfig)
     */
    public void init(PseudoServerConfiguration configuration)
            throws PseudoServerException {
        Ini.Section cfgSection = configuration
                .getAdminAccessControlListConfiguration();
        String filename = cfgSection.get(ACLFILE_CONFIG_PARAM);
        log.info("{}={}", ACLFILE_CONFIG_PARAM, filename);
        if (filename == null) {
            throw new PseudoServerException("Configuration parameter "
                    + ACLFILE_CONFIG_PARAM + " is not defined");
        }

        // load the XML file
        aclXMLConfiguration_ = createACLXMLConfiguration(filename);
        log.info("XMLFileAccessControlList successfully initialized");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.slcs.acl.AccessControlList#isAuthorized(java.util.List)
     */
    public boolean isAuthorized(List<Attribute> userAttributes) {
        if (log.isDebugEnabled()) {
            log.debug("userAttributes=" + userAttributes);
        }
        boolean authorized = false;

        if (accessControlRules_ == null) {
            log.debug("accessControlRules_ was empty, creating a new one");
            // create the access control rules list
            accessControlRules_ = createACLAccessControlRules(
                    aclXMLConfiguration_,
                    (AttributeDefinitions) context
                            .getBean("pseudonymity.AttributeDefinitions"));
            log.debug("new accessControlRules_ successfully created");
        }

        Iterator<AccessControlRule> rules = accessControlRules_.iterator();
        while (!authorized && rules.hasNext()) {
            AccessControlRule rule = (AccessControlRule) rules.next();
            List<Attribute> ruleAttributes = rule.getAttributes();
            if (log.isDebugEnabled()) {
                log.debug("checking rule:" + rule);
            }
            // only rule attrs know if they are caseSensitive or not...
            if (userAttributes.containsAll(ruleAttributes)) {
                authorized = true;
                log.info("User authorized by rule: " + rule);
            }
        }

        if (!authorized) {
            log.warn("User not authorized: " + userAttributes);
        }

        return authorized;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.slcs.acl.AccessControlList#shutdown()
     */
    public void shutdown() {
    }

    /**
     * Loads the ACL XML FileConfiguration.
     * 
     * @param filename
     *            The ACL XML filename to load.
     * @return The FileConfiguration object.
     * @throws SLCSConfigurationException
     *             If an configration error occurs while loading the file.
     */
    static private XMLConfiguration createACLXMLConfiguration(String filename)
            throws PseudoServerException {
        XMLConfiguration config = null;
        try {
            log.info("XMLConfiguration file=" + filename);
            config = new XMLConfiguration(filename);
            if (log.isDebugEnabled()) {
                File configFile = config.getFile();
                log.debug("XMLConfiguration file="
                        + configFile.getAbsolutePath());
            }
        } catch (ConfigurationException e) {
            log.error("Failed to create XMLConfiguration: " + filename, e);
            throw new PseudoServerException(
                    "Failed to create XMLConfiguration: " + filename, e);
        }
        return config;
    }

    /**
     * Creates a list of {@link AccessControlRule}s loaded from the
     * {@link FileConfiguration}.
     * 
     * @param config
     *            The ACL FileConfiguration object
     * @return A {@link List} of {@link AccessControlRule}s
     */
    static private List<AccessControlRule> createACLAccessControlRules(
            FileConfiguration config, AttributeDefinitions attributeDefinitions) {
        List<AccessControlRule> accessControlRules = new LinkedList<AccessControlRule>();
        // list all rules
        int i = 0;
        while (true) {
            String rulePrefix = "AccessControlRule(" + i + ")";
            i++;
            // get the name and id of the rule
            String ruleGroup = config.getString(rulePrefix + "[@group]");
            if (ruleGroup == null) {
                log.debug("{}: no more rules", rulePrefix);
                // no more ACL rule to read, exit while loop
                break;
            }
            int ruleId = config.getInt(rulePrefix + "[@id]");
            // create an empty rule
            AccessControlRule rule = new AccessControlRule(ruleId, ruleGroup);
            // get the attributes name-value for the rule
            List<?> attributeNames = config.getList(rulePrefix
                    + ".Attribute[@name]");
            log.debug("Found {} attributes for the rule", attributeNames.size());
            if (attributeNames.isEmpty()) {
                log.error(rulePrefix + ": no attribute in rule, skipping...");
                // error, skipping
                continue;
            }

            List<?> attributeValues = config.getList(rulePrefix + ".Attribute");
            for (int j = 0; j < attributeNames.size(); j++) {
                String name = (String) attributeNames.get(j);
                String value = (String) attributeValues.get(j);
                Attribute attribute = attributeDefinitions.createAttribute(
                        name, value);
                // add attribute to the rule
                rule.addAttribute(attribute);
            }
            // add the rule to the list
            if (log.isDebugEnabled()) {
                log.debug("adding rule in ACL: " + rule);
            }
            accessControlRules.add(rule);

        } // while

        return accessControlRules;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.springframework.context.ApplicationContextAware#setApplicationContext
     * (org.springframework.context.ApplicationContext)
     */
    public void setApplicationContext(ApplicationContext context)
            throws BeansException {
        log.debug("Setting the Spring application context");
        this.context = context;
    }

}
