/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.server.aaclient.impl;

import java.io.IOException;
import java.security.KeyStoreException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

import javax.security.auth.x500.X500Principal;

import org.apache.axis.configuration.SimpleProvider;
import org.glite.pseudo.server.PseudoServerException;
import org.glite.pseudo.server.aaclient.AAClient;
import org.glite.pseudo.server.aaclient.AAClientException;
import org.glite.pseudo.server.caclient.impl.AbstractSSLClient;
import org.glite.pseudo.server.config.PseudoServerConfiguration;
import org.glite.security.voms.service.certificates.VOMSCertificates;
import org.glite.security.voms.service.certificates.VOMSCertificatesServiceLocator;
import org.ini4j.Ini;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import eu.emi.security.authn.x509.X509CertChainValidator;
import eu.emi.security.authn.x509.X509Credential;
import eu.emi.security.authn.x509.impl.OpensslCertChainValidator;
import eu.emi.security.authn.x509.impl.PEMCredential;
import eu.emi.security.authn.x509.impl.X500NameUtils;

/**
 * <p>
 * VOMSClient registers the certificates to the VOMS server using VOMS-Admin Web
 * service interface
 * </p>
 */
public class VOMSClient extends AbstractSSLClient implements AAClient {

    /** Logging */
    private static final Logger log = LoggerFactory.getLogger(VOMSClient.class);
    
    /** The directory for the CA certificates. */
    public static final String CA_CERTIFICATE_DIRECTORY = "/etc/grid-security/certificates";

    /** VOMS Certificate Service */
    private VOMSCertificates vomsCertService;

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.aaclient.AAClient#registerCertificate(java.security
     * .cert.X509Certificate, java.security.cert.X509Certificate)
     */
    public void registerCertificate(X509Certificate original,
            X509Certificate alias) throws AAClientException {
        X500Principal originalDN;
        X500Principal caDN;
        X500Principal aliasDN;
		try {
			originalDN = X500NameUtils.getX500Principal(original.getSubjectDN().toString());
	        caDN = X500NameUtils.getX500Principal(original.getIssuerDN().toString());
	        aliasDN = X500NameUtils.getX500Principal(alias.getSubjectDN().toString());
		} catch (IOException e) {
			log.debug("Could not convert a DN into an X500Principal", e);
			throw new AAClientException("Could not convert a DN into an X500 Principal", e);
		}

        log.info("Registering " + aliasDN + " as an alias to " + originalDN);

        try {
            log.debug("Generating VOMS X509Certificate");
            org.glite.security.voms.service.certificates.X509Certificate newCert = new org.glite.security.voms.service.certificates.X509Certificate(
                    alias.getEncoded(), alias.getSerialNumber().longValue(),
                    alias.getIssuerDN().toString(), alias.getNotAfter()
                            .toString(), aliasDN.toString());
/*            org.glite.wsdl.services.org_glite_security_voms_service_certificates.X509Certificate newCert = org.glite.wsdl.services.org_glite_security_voms_service_certificates.X509Certificate.Factory.newInstance();
            newCert.setIssuer(alias.getIssuerDN().toString());
            newCert.setId(alias.getSerialNumber().longValue());
            newCert.setNotAfter(alias.getNotAfter().toString());
            newCert.setSubject(aliasDN.toString());
            newCert.setBytes(new Base64Binary(alias.getEncoded()));
            org.glite.wsdl.services.org_glite_security_voms_service_certificates.X509Certificate.Factory.newInstance();*/
            this.vomsCertService.addCertificate(originalDN.toString(), caDN.toString(), newCert);
        } catch (Exception e) {
            throw new AAClientException(
                    "Could not add new certificate to the VOMS certificate service!",
                    e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.glite.pseudo.server.PseudoServerComponent#init(org.glite.pseudo.server
     * .config.PseudoServerConfiguration)
     */
    public void init(PseudoServerConfiguration configuration)
            throws PseudoServerException {
        log.debug("Initializing VOMSClient");
        Ini.Section cfgSection = configuration.getAAClientConfiguration();
        super.init(cfgSection);
        if (keyFile == null) {
            throw new PseudoServerException(
                    "Private key for client authentication is not configured!");
        } else {
            log.debug("Using {} as a private key for client authentication.",
                    keyFile);
        }

        if (certFile == null) {
            throw new PseudoServerException(
                    "Client authentication certificate is not configured!");
        } else {
            log.debug("Using {} as a certificate for client authentication.",
                    certFile);
        }

        X509Credential credential;
        try {
        	if (keyPasswd == null) {
        		log.debug("Password for the private key is not configured, the private key should be plaintext.");
        		credential = new PEMCredential(keyFile, certFile, null);
        	} else {
        		log.debug("Using a password for decrypting the private key.");
        		credential = new PEMCredential(keyFile, certFile, keyPasswd.toCharArray());
        	}
		} catch (KeyStoreException e) {
			throw new PseudoServerException("Could not initialize the private key!", e);
		} catch (CertificateException e) {
			throw new PseudoServerException("Could not initialize the X.509 certificate!", e);
		} catch (IOException e) {
			throw new PseudoServerException("Could not read the private key and/or certificate file!", e);
		}
  
        X509CertChainValidator validator = new OpensslCertChainValidator(CA_CERTIFICATE_DIRECTORY);

        try {
            SimpleProvider provider = SSLConfigSender.getTransportProvider(credential, validator);
            VOMSCertificatesServiceLocator locator = new VOMSCertificatesServiceLocator(
                    provider);
            locator.setVOMSCertificatesEndpointAddress(endpoint);
            this.vomsCertService = locator.getVOMSCertificates();
        } catch (Exception e) {
            throw new PseudoServerException(
                    "Error during VOMS certificate service client initialization!",
                    e);
        }

        log.info("VOMS certificate service client successfully initialized.");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.glite.pseudo.server.caclient.impl.AbstractSSLClient#shutdown()
     */
    public void shutdown() {
        super.shutdown();

    }

}
