# Copyright (c) Members of the EMI Collaboration. 2011.
# See http://eu-emi.eu/partners/ for details on the copyright holders.
# For license conditions see http://www.apache.org/licenses/LICENSE-2.0
#

#   chkconfig: 345 97 97
#   description:  Pseudonymity server startup script
#   processname: pseudo-service

set -e

prog="$(basename $0)"

if [ -r "/etc/sysconfig/pseudonymity-service" ]; then
        source /etc/sysconfig/pseudonymity-service
fi

if [ -z $PSEUDO_SERVER_HOME ]; then
        PSEUDO_SERVER_HOME="$(cd "${0%/*}/.." && pwd)"
fi

if [ ! -r $PSEUDO_SERVER_HOME/conf/pseudonymity-server.ini ]; then
        echo "File $PSEUDO_SERVER_HOME/conf/pseudonymity-server.ini doesn't exist or is not readable!"
        exit 1
fi 
        
if [ -z $PSEUDO_RUN_FILE ]; then
        PSEUDO_RUN_FILE=$PSEUDO_SERVER_HOME/.pseudonymity-service.pid
fi

. $PSEUDO_SERVER_HOME/sbin/pseudo-env.sh

pre_checks(){
        check_openssl
        check_certificates
}

success () {
        if [ ! -z "$1" ]; then
                echo "$1"
        fi
        exit 0
}

failure () {
        if [ ! -z "$1" ]; then
                echo "$1"
        fi
        exit 1
}

pseudo_pid(){

        if test ! -f $PSEUDO_RUN_FILE; then
                echo "No pid file found for $prog!"
                failure
        fi
        
        PSEUDO_PID=`head -n 1 $PSEUDO_RUN_FILE`
        
}

kill_pseudo_proc(){

        status
        if [ $? -eq 0 ]; then
                ## pseudo process is running
                pid=`head -1 $PSEUDO_RUN_FILE`
                
                ## Use shutdown service hook
                $PSEUDO_SHUTDOWN_CMD
                
                if [ $? -ne 0 ]; then
                        echo "Error shutting down Pseudonymity Service! Will kill the process..."
                        
                        kill -9 $pid
                        
                        if [ $? -ne 0 ]; then
                                failure "Error killing the Pseudonymity Service... maybe you don't have the permissions to kill it!"
                        else
                                ## remove pid file
                                rm $PSEUDO_RUN_FILE
                        fi
                        
                else
                        ## remove pid file
                        rm $PSEUDO_RUN_FILE
                fi
        else
                failure "Pseudonymity server is not running!"
        fi 
}
        
status(){

        if [ -f $PSEUDO_RUN_FILE ]; then
                pid=`head -1 $PSEUDO_RUN_FILE`
                ps -p $pid >/dev/null 2>&1
                
                if [ $? -ne 0 ]; then
                        echo "Pseudonymity Service not running...removing stale pid file"
                        rm $PSEUDO_RUN_FILE
                        return 1
                else
                        return 0
                fi
        else
                ## No PID file found, check that there is no PSEUDO running 
                ## without PID...
                pid=`ps -efww | grep 'java.*PseudonymityService' | grep -v grep | awk '{print $2}'`
                if [ -z $pid ]; then
                        return 1
                else
                        echo "A Pseudonymity Service is running, but no pid file found! Will recreate the pid file"
                        echo $pid > $PSEUDO_RUN_FILE
                        if [ $? -ne 0 ]; then
                                failure "Error creating pid file for running Pseudonymity Service!"
                        fi
                        return 0
                fi
        fi
}


start(){

        # echo -n "Starting $prog: "
        
        status && failure "Pseudonymity Service already running..."
        
        $PSEUDO_CMD &
        
        if [ $? -eq 0 ]; then
                echo "$!" > $PSEUDO_RUN_FILE;
                status || failure "Pseudonymity Service not running after being just started!"
                success 
        else
                failure "failed!"
        fi
                
}

restart(){
        
        # echo -n "Restarting $prog: "
        kill_pseudo_proc && (rm -f $PSEUDO_RUN_FILE; sleep 5; start) || failure "Error restarting Pseudonymity Service process!"

}
stop(){
        # echo -n "Stopping $prog: "
        kill_pseudo_proc && (rm -f $PSEUDO_RUN_FILE; success "Ok.") || failure "Error killing Pseudonymity Service process!"
}

case "$1" in
        start)
                start
                ;;
        
        stop)
                stop
                ;;
        
        status)
                status && success "Pseudonymity Service running!" || failure "Pseudonymity Service not running!"
                ;;
        
        restart)
                restart
                ;;
        *)
                echo "Usage: $0 {start|stop|status}"
                RETVAL=1
                ;;
esac
exit $RETVAL