/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.common.util;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;

import org.apache.commons.lang.SystemUtils;

/**
 * Some static utility methods used by both the UI and the server.
 */
public class Util {

    /**
     * Sets permissions on a given file. The permissions are set using the
     * <i>chmod</i> command and will only work on Unix machines. Chmod command
     * must be in the path.
     * 
     * @param file
     *            the file to set the permissions of.
     * @param mode
     *            the Unix style permissions.
     * @return true, if change was successful, otherwise false. It can return
     *         false, in many instances, e.g. when file does not exits, when
     *         chmod is not found, or other error occurs.
     */
    public static boolean setFilePermissions(File file, int mode) {
        String filename= file.getPath();
        if (!file.exists()) {
            try {
                file.createNewFile();
            } catch (IOException e) {
                // ignored e.printStackTrace();
            }
        }
        // only on Unix
        if (!SystemUtils.IS_OS_WINDOWS) {
            Runtime runtime= Runtime.getRuntime();
            String[] cmd= new String[] { "chmod", String.valueOf(mode), filename };
            try {
                Process process= runtime.exec(cmd);
                return (process.waitFor() == 0) ? true : false;
            } catch (Exception e) {
                return false;
            }
        }
        // on Windows always return true
        else {
            return true;
        }       
    }
    
    /**
     * @param utf8
     *            The strange encoded UTF8 string.
     * @return a Java unicode string or <code>null</code> if the conversion
     *         failed.
     */
    static public String convertUTF8ToUnicode(String utf8) throws UnsupportedEncodingException {
        if (utf8 == null) {
            return null;
        }
        String unicode= null;
        // get the received bytes as ISO-Latin1 encoded
        byte [] latin1= utf8.getBytes("ISO-8859-1");
        // but interpret them as UTF-8
        unicode= new String(latin1, "UTF-8");
        return unicode;
    }
    
    /**
     * Removes preceeding or proceeding whitespace from a string or return null if the string is null or of zero length
     * after trimming (i.e. if the string only contained whitespace).
     * 
     * @param s the string to trim
     * 
     * @return the trimmed string or null
     */
    public static String safeTrimOrNullString(String s) {
        if (s != null) {
            String sTrimmed = s.trim();
            if (sTrimmed.length() > 0) {
                return sTrimmed;
            }
        }
        return null;
    }
    
    /**
     * <p>
     * Checks whether two char-arrays match.
     * </p>
     * 
     * @param first
     *            the first array
     * @param second
     *            the other array
     * @return <code>true</code> if match, <code>false</code> otherwise.
     */
    public static boolean matchCharArrays(char[] first, char[] second) {
        if (first == null || second == null || first.length != second.length) {
            return false;
        }
        for (int i = 0; i < first.length; i++) {
            if (first[i] != second[i]) {
                return false;
            }
        }
        return true;
    }
    
}
