/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.common.pki.bouncycastle;

import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.Security;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Wrapper class for the BouncyCastle KeyPairGenerator. Uses a BouncyCastle
 * org.bouncycastle.jce.provider.JDKKeyPairGenerator key pair generator as
 * delegate.
 * 
 * Imported from <code>org.glite.slcs.pki.bouncycastle.KeyPairGenerator</code>.
 * 
 * @author Valery Tschopp &lt;tschopp@switch.ch&gt;
 */
public class KeyPairGenerator {
	
    /** Logging */
	private final Logger log = LoggerFactory.getLogger(KeyPairGenerator.class);
	
    /**
     * Sets BouncyCastle security provider as boot time
     */
    static {
        // add only once
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * BouncyCaslte KeyPairGenerator delegate
     */
    private java.security.KeyPairGenerator generator_ = null;

    /**
     * Construtor.
     * 
     * @param algorithm
     * @throws GeneralSecurityException
     */
    public KeyPairGenerator(String algorithm) throws GeneralSecurityException {
        generator_ = java.security.KeyPairGenerator.getInstance(algorithm, BouncyCastleProvider.PROVIDER_NAME);
    }

    /**
     * Initializes the generator to generate key pair of the given size.
     * 
     * @param keysize
     *            The size of the keys to generate.
     */
    public void initialize(int keysize) {
    	log.debug("Initializing a keypair with keysize={}", keysize);
        generator_.initialize(keysize);
    }

    /**
     * Generates a new {@link KeyPair}
     * 
     * @return The new {@link KeyPair}
     */
    public KeyPair generateKeyPair() {
    	log.debug("Generating a new keypair");
        return generator_.generateKeyPair();
    }


}
