/**
 * Copyright (c) Members of the EMI Collaboration. 2011.
 * See http://eu-emi.eu/partners/ for details on the copyright holders.
 * For license conditions see http://www.apache.org/licenses/LICENSE-2.0
 */
package org.glite.pseudo.common.pki;

import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.Vector;

import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.DERObjectIdentifier;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.x509.ExtendedKeyUsage;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.asn1.x509.KeyUsage;
import org.bouncycastle.asn1.x509.PolicyInformation;
import org.bouncycastle.asn1.x509.X509Extension;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * X.509 certificate extensions factory
 *
 * Imported from org.glite.slcs.pki.CertificateExtensionFactory
 *
 * @author Valery Tschopp <tschopp@switch.ch>
 */
public class CertificateExtensionFactory {
	
	/** Logging */
	private final static Logger log = LoggerFactory.getLogger(CertificateExtensionFactory.class);
	
    /**
     * Creates a non-critical CertificateExtension. The id can be the OID or the name as
     * defined below. The values is a comma separated list of value(s)
     * <p>
     * Valid names and values:
     * <ul>
     * <li>KeyUsage
     * <ul>
     * <li>DigitalSignature
     * <li>NonRepudiation
     * <li>KeyEncipherment
     * <li>DataEncipherment
     * <li>KeyAgreement
     * <li>KeyCertSign
     * <li>CRLSign
     * <li>EncipherOnly
     * <li>DecipherOnly
     * </ul>
     * <li>ExtendedKeyUsage
     * <ul>
     * <li>AnyExtendedKeyUsage
     * <li>ServerAuth
     * <li>ClientAuth
     * <li>CodeSigning
     * <li>EmailProtection
     * <li>IPSecEndSystem
     * <li>IPSecTunnel
     * <li>IPSecUser
     * <li>OCSPSigning
     * <li>Smartcardlogon
     * </ul>
     * <li>CertificatePolicies
     * <ul>
     * <li>The policy OID(s)
     * </ul>
     * <li>SubjectAltName
     * <ul>
     * <li>email:EMAIL_ADDRESS
     * <li>dns:HOSTNAME
     * </ul>
     * </ul>
     * <p>
     * Example:
     * <pre>
     * CertificateExtension keyUsageExtension = 
     *       CertificateExtensionFactory.createCertificateExtension("KeyUsage", "DigitalSignature,KeyEncipherment");
     * CertificateExtension subjectAltNameExtension = 
     *       CertificateExtensionFactory.createCertificateExtension("SubjectAltName", "email:john.doe@example.com,dns:www.exmaple.com");
     * </pre>
     * 
     * @param id
     *            The name or the OID of the extension.
     * @param values
     *            A comma separated list of extension value(s).
     * @return The corresponding CertificateExtension or <code>null</code> if
     *         the id (name or oid) is not supported.
     */
    public static CertificateExtension createCertificateExtension(String id,
            String values) {
    	log.debug("Creating a new non-critical CertificateExtension id={}, values={}", id, values);
    	return createCertificateExtension(id, values, false);
    }
	
    /**
     * Creates a CertificateExtension. The id can be the OID or the name as
     * defined below. The values is a comma separated list of value(s)
     * <p>
     * Valid names and values:
     * <ul>
     * <li>KeyUsage
     * <ul>
     * <li>DigitalSignature
     * <li>NonRepudiation
     * <li>KeyEncipherment
     * <li>DataEncipherment
     * <li>KeyAgreement
     * <li>KeyCertSign
     * <li>CRLSign
     * <li>EncipherOnly
     * <li>DecipherOnly
     * </ul>
     * <li>ExtendedKeyUsage
     * <ul>
     * <li>AnyExtendedKeyUsage
     * <li>ServerAuth
     * <li>ClientAuth
     * <li>CodeSigning
     * <li>EmailProtection
     * <li>IPSecEndSystem
     * <li>IPSecTunnel
     * <li>IPSecUser
     * <li>OCSPSigning
     * <li>Smartcardlogon
     * </ul>
     * <li>CertificatePolicies
     * <ul>
     * <li>The policy OID(s)
     * </ul>
     * <li>SubjectAltName
     * <ul>
     * <li>email:EMAIL_ADDRESS
     * <li>dns:HOSTNAME
     * </ul>
     * </ul>
     * <p>
     * Example:
     * <pre>
     * CertificateExtension keyUsageExtension = 
     *       CertificateExtensionFactory.createCertificateExtension("KeyUsage", "DigitalSignature,KeyEncipherment", true);
     * CertificateExtension subjectAltNameExtension = 
     *       CertificateExtensionFactory.createCertificateExtension("SubjectAltName", "email:john.doe@example.com,dns:www.exmaple.com", false);
     * </pre>
     * 
     * @param id
     *            The name or the OID of the extension.
     * @param values
     *            A comma separated list of extension value(s).
     * @param critical
     *            If the extension should be critical or not.
     * @return The corresponding CertificateExtension or <code>null</code> if
     *         the id (name or oid) is not supported.
     */
    public static CertificateExtension createCertificateExtension(String id,
            String values, boolean critical) {
    	log.debug("Creating a new CertificateExtension id={}, values={}", id, values);
        if (log.isDebugEnabled()) {
            log.debug("id:" + id + " value(s):" + values);
        }
        if (id.equals(X509Extension.keyUsage.getId())
                || id.equalsIgnoreCase("KeyUsage")) {
            // parse the comma separated list of key usage
            int usage= 0;
            StringTokenizer st= new StringTokenizer(values, ",");
            while (st.hasMoreElements()) {
                String keyUsage= (String) st.nextElement();
                keyUsage= keyUsage.trim();

                if (keyUsage.equalsIgnoreCase("DigitalSignature")) {
                    usage+= KeyUsage.digitalSignature;
                }
                else if (keyUsage.equalsIgnoreCase("NonRepudiation")) {
                    usage+= KeyUsage.nonRepudiation;
                }
                else if (keyUsage.equalsIgnoreCase("KeyEncipherment")) {
                    usage+= KeyUsage.keyEncipherment;
                }
                else if (keyUsage.equalsIgnoreCase("DataEncipherment")) {
                    usage+= KeyUsage.dataEncipherment;
                }
                else if (keyUsage.equalsIgnoreCase("KeyAgreement")) {
                    usage+= KeyUsage.keyAgreement;
                }
                else if (keyUsage.equalsIgnoreCase("KeyCertSign")) {
                    usage+= KeyUsage.keyCertSign;
                }
                else if (keyUsage.equalsIgnoreCase("CRLSign")) {
                    usage+= KeyUsage.cRLSign;
                }
                else if (keyUsage.equalsIgnoreCase("EncipherOnly")) {
                    usage+= KeyUsage.encipherOnly;
                }
                else if (keyUsage.equalsIgnoreCase("DecipherOnly")) {
                    usage+= KeyUsage.decipherOnly;
                }
                else {
                    log.error("Unknown KeyUsage: " + keyUsage);
                }

            }
            return createKeyUsageExtension(usage, values, critical);
        }
        else if (id.equals(X509Extension.extendedKeyUsage.getId())
                || id.equalsIgnoreCase("ExtendedKeyUsage")) {
            // value is a comma separated list of keyPurpose
            Vector<KeyPurposeId> keyPurposeIds= new Vector<KeyPurposeId>();
            StringTokenizer st= new StringTokenizer(values, ",");
            while (st.hasMoreElements()) {
                String keyPurpose= (String) st.nextElement();
                keyPurpose= keyPurpose.trim();
                if (keyPurpose.equalsIgnoreCase("AnyExtendedKeyUsage")) {
                    keyPurposeIds.add(KeyPurposeId.anyExtendedKeyUsage);
                }
                else if (keyPurpose.equalsIgnoreCase("ServerAuth")) {
                    keyPurposeIds.add(KeyPurposeId.id_kp_serverAuth);
                }
                else if (keyPurpose.equalsIgnoreCase("ClientAuth")) {
                    keyPurposeIds.add(KeyPurposeId.id_kp_clientAuth);
                }
                else if (keyPurpose.equalsIgnoreCase("CodeSigning")) {
                    keyPurposeIds.add(KeyPurposeId.id_kp_codeSigning);
                }
                else if (keyPurpose.equalsIgnoreCase("EmailProtection")) {
                    keyPurposeIds.add(KeyPurposeId.id_kp_emailProtection);
                }
                else if (keyPurpose.equalsIgnoreCase("IPSecEndSystem")) {
                    keyPurposeIds.add(KeyPurposeId.id_kp_ipsecEndSystem);
                }
                else if (keyPurpose.equalsIgnoreCase("IPSecTunnel")) {
                    keyPurposeIds.add(KeyPurposeId.id_kp_ipsecTunnel);
                }
                else if (keyPurpose.equalsIgnoreCase("IPSecUser")) {
                    keyPurposeIds.add(KeyPurposeId.id_kp_ipsecUser);
                }
                else if (keyPurpose.equalsIgnoreCase("TimeStamping")) {
                    keyPurposeIds.add(KeyPurposeId.id_kp_timeStamping);
                }
                else if (keyPurpose.equalsIgnoreCase("OCSPSigning")) {
                    keyPurposeIds.add(KeyPurposeId.id_kp_OCSPSigning);
                }
                else if (keyPurpose.equalsIgnoreCase("Smartcardlogon")) {
                    keyPurposeIds.add(KeyPurposeId.id_kp_smartcardlogon);
                }
                else {
                    log.error("Unknown ExtendedKeyUsage: " + keyPurpose);
                }
            }
            return createExtendedKeyUsageExtension(keyPurposeIds, values, critical);
        }
        else if (id.equals(X509Extension.certificatePolicies.getId())
                || id.equalsIgnoreCase("CertificatePolicies")) {
            // values is a comma separated list of policyOIDs
            Vector<String> policyOIDs= new Vector<String>();
            StringTokenizer st= new StringTokenizer(values, ",");
            while (st.hasMoreElements()) {
                String policyOID= (String) st.nextElement();
                policyOID= policyOID.trim();
                policyOIDs.add(policyOID);
            }
            return createCertificatePoliciesExtension(policyOIDs, values, critical);
        }
        else if (id.equals(X509Extension.subjectAlternativeName.getId())
                || id.equalsIgnoreCase("SubjectAltName")) {
            // values is a comma separated list of altername names prefixed with
            // the type (email: or dns:)
            Vector<String> typedSubjectAltNames= new Vector<String>();
            StringTokenizer st= new StringTokenizer(values, ",");
            while (st.hasMoreElements()) {
                String typedAltName= (String) st.nextElement();
                typedAltName= typedAltName.trim();
                typedSubjectAltNames.add(typedAltName);
            }
            return createSubjectAltNameExtension(typedSubjectAltNames, values, critical);
        }
        log.error("Unsupported CertificateExtension: " + id);
        return null;
    }

    /**
     * 
     * @param keyPurposeIds
     * @param keyPurposeNames
     * @return
     */
    static protected CertificateExtension createExtendedKeyUsageExtension(
            Vector<KeyPurposeId> keyPurposeIds, String keyPurposeNames, boolean critical) {
        ExtendedKeyUsage extendedKeyUsage= new ExtendedKeyUsage(keyPurposeIds);
        X509Extension extendedKeyUsageExtension= new X509Extension(critical,
                                                                   new DEROctetString(extendedKeyUsage));
        return new CertificateExtension(X509Extension.extendedKeyUsage,
                                        "ExtendedKeyUsage",
                                        extendedKeyUsageExtension,
                                        keyPurposeNames,
                                        critical);
    }

    /**
     * 
     * @param keyPurposeId
     * @param keyPurposeName
     * @return
     */
    static protected CertificateExtension createExtendedKeyUsageExtension(
            KeyPurposeId keyPurposeId, String keyPurposeName, boolean critical) {
        DERSequence keyPurposeIds= new DERSequence(keyPurposeId);
        ExtendedKeyUsage extendedKeyUsage= new ExtendedKeyUsage(keyPurposeIds);
        X509Extension extendedKeyUsageExtension= new X509Extension(critical,
                                                                   new DEROctetString(extendedKeyUsage));
        return new CertificateExtension(X509Extension.extendedKeyUsage,
                                        "ExtendedKeyUsage",
                                        extendedKeyUsageExtension,
                                        keyPurposeName,
                                        critical);
    }

    /**
     * Creates a RFC882 Subject Alternative Name: email:johndoe@example.com
     * extension.
     * 
     * @param emailAddress
     *            The email address to be included as alternative name.
     * @return The subject alternative name CertificateExtension.
     */
    static protected CertificateExtension createSubjectAltNameExtension(
            String emailAddress, boolean critical) {
        GeneralName subjectAltName= new GeneralName(GeneralName.rfc822Name,
                                                    emailAddress);
        GeneralNames subjectAltNames= new GeneralNames(subjectAltName);
        X509Extension subjectAltNameExtension= new X509Extension(critical,
                                                                 new DEROctetString(subjectAltNames));
        return new CertificateExtension(X509Extension.subjectAlternativeName,
                                        "SubjectAltName",
                                        subjectAltNameExtension,
                                        emailAddress,
                                        critical);

    }

    /**
     * 
     * @param prefixedAltNames
     * @param values
     * @return
     */
    static protected CertificateExtension createSubjectAltNameExtension(
            Vector<String> prefixedAltNames, String values, boolean critical) {
        ASN1EncodableVector altNames= new ASN1EncodableVector();
        Enumeration<String> typeAndNames= prefixedAltNames.elements();
        while (typeAndNames.hasMoreElements()) {
            String typeAndName= typeAndNames.nextElement();
            typeAndName= typeAndName.trim();
            if (typeAndName.startsWith("email:")) {
                String emailAddress= typeAndName.substring("email:".length());
                GeneralName altName= new GeneralName(GeneralName.rfc822Name,
                                                     emailAddress);
                altNames.add(altName);

            }
            else if (typeAndName.startsWith("dns:")) {
                String hostname= typeAndName.substring("dns:".length());
                GeneralName altName= new GeneralName(GeneralName.dNSName,
                                                     hostname);
                altNames.add(altName);
            }
            else {
                //log.error("Unsupported subjectAltName: " + typeAndName);
            }
        }
        DERSequence subjectAltNames= new DERSequence(altNames);
        GeneralNames generalNames= new GeneralNames(subjectAltNames);
        X509Extension subjectAltNameExtension= new X509Extension(critical,
                                                                 new DEROctetString(generalNames));
        return new CertificateExtension(X509Extension.subjectAlternativeName,
                                        "SubjectAltName",
                                        subjectAltNameExtension,
                                        values,
                                        critical);

    }

    /**
     * Creates a Cerificate Policies: policyOID extension with the given policy
     * OID.
     * 
     * @param policyOID
     *            The policy OID (2.16.756.1.2.*)
     * @return The certificate policies CertificateExtension.
     */
    static protected CertificateExtension createCertificatePoliciesExtension(
            String policyOID, boolean critical) {
        DERObjectIdentifier policyIdentifier= new DERObjectIdentifier(policyOID);
        PolicyInformation policyInformation= new PolicyInformation(policyIdentifier);
        DERSequence certificatePolicies= new DERSequence(policyInformation);
        X509Extension certificatePoliciesExtension= new X509Extension(critical,
                                                                      new DEROctetString(certificatePolicies));
        return new CertificateExtension(X509Extension.certificatePolicies,
                                        "CertificatePolicies",
                                        certificatePoliciesExtension,
                                        policyOID,
                                        critical);
    }

    /**
     * 
     * @param policyOIDs
     * @param values
     * @return
     */
    static protected CertificateExtension createCertificatePoliciesExtension(
            Vector<String> policyOIDs, String values, boolean critical) {
        ASN1EncodableVector policyInformations= new ASN1EncodableVector();
        Enumeration<String> pOids= policyOIDs.elements();
        while (pOids.hasMoreElements()) {
            String policyOid= pOids.nextElement();
            DERObjectIdentifier policyIdentifier= new DERObjectIdentifier(policyOid);
            PolicyInformation policyInformation= new PolicyInformation(policyIdentifier);
            policyInformations.add(policyInformation);

        }
        DERSequence certificatePolicies= new DERSequence(policyInformations);
        X509Extension certificatePoliciesExtension= new X509Extension(critical,
                                                                      new DEROctetString(certificatePolicies));
        return new CertificateExtension(X509Extension.certificatePolicies,
                                        "CertificatePolicies",
                                        certificatePoliciesExtension,
                                        values,
                                        critical);
    }

    /**
     * Creates a Key Usage extension for the given usage. This extension is
     * critical.
     * 
     * @param usage
     *            The usage is the sum of all KeyUsage values.
     * @param value
     *            The formal value of the usage. Example:
     *            KeyEncipherment,DigitalSignature
     * @return The KeyUsage certificate extension.
     * @see org.bouncycastle.asn1.x509.KeyUsage
     */
    static protected CertificateExtension createKeyUsageExtension(int usage,
            String value, boolean critical) {
        KeyUsage keyUsage= new KeyUsage(usage);
        X509Extension keyUsageExtension= new X509Extension(critical,
                                                           new DEROctetString(keyUsage));
        return new CertificateExtension(X509Extension.keyUsage,
                                        "KeyUsage",
                                        keyUsageExtension,
                                        value,
                                        critical);
    }

    /**
     * Do not allow instantiation of the factory.
     */
    private CertificateExtensionFactory() {
    }


}
