/*
 * Copyright (c) Members of the EGEE Collaboration. 2006-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.glite.authz.pep.obligation.eesmap;

import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import org.glite.authz.common.context.DecisionRequestContext;
import org.glite.authz.common.context.DecisionRequestContextHelper;
import org.glite.authz.common.model.Attribute;
import org.glite.authz.common.model.AttributeAssignment;
import org.glite.authz.common.model.Environment;
import org.glite.authz.common.model.Obligation;
import org.glite.authz.common.model.Request;
import org.glite.authz.common.model.Response;
import org.glite.authz.common.model.Result;
import org.glite.authz.common.model.util.DeepCopy;
import org.glite.authz.pep.obligation.BaseObligationHandler;
import org.glite.authz.pep.obligation.ObligationHandler;
import org.glite.authz.pep.obligation.ObligationProcessingException;

import org.opensaml.Configuration;
import org.opensaml.saml2.core.Assertion;
import org.opensaml.saml2.core.Statement;
import org.opensaml.ws.soap.client.http.HttpClientBuilder;
import org.opensaml.ws.soap.client.http.HttpSOAPClient;
import org.opensaml.ws.soap.common.SOAPException;
import org.opensaml.ws.soap.soap11.Envelope;
import org.opensaml.xacml.ctx.RequestType;
import org.opensaml.xacml.ctx.ResponseType;
import org.opensaml.xacml.profile.saml.XACMLAuthzDecisionStatementType;
import org.opensaml.xml.io.MarshallingException;
import org.opensaml.xml.parse.BasicParserPool;
import org.opensaml.xml.security.SecurityException;
import org.opensaml.xml.util.XMLHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

/**
 * TODO documentation
 */
public class EESObligationHandler extends BaseObligationHandler implements
        ObligationHandler {
    /** Logger. */
    private final Logger log= LoggerFactory.getLogger(EESObligationHandler.class);

    /** OH entityID */
    private String entityId;

    /** EES endpoint URL */
    private String eesEndpoint;

    /** TODO DOC */
    static final String SPECIAL_EES_OBLIGATION_LIST_ATTR_ID= "http://authz-interop.org/xacml/attribute/trans-obligation";

    /** SOAP client to talk with EES */
    private HttpSOAPClient soapClient;

    /**
     * Constructor.
     * 
     * @param name
     *            The obligation handler name
     * @param obligationId
     *            The ID of the handled obligation
     * @param endPoint
     *            The EES endpoint URL
     */
    public EESObligationHandler(String name, String obligationId,
            String endPoint) {
        super(name, obligationId);

        this.eesEndpoint= endPoint;

        // create a SOAP client
        BasicParserPool parserPool= new BasicParserPool();
        parserPool.setNamespaceAware(true);
        HttpClientBuilder clientBuilder= new HttpClientBuilder();
        // TODO: And if SOAP client need HTTPS support??? 
        soapClient= new HttpSOAPClient(clientBuilder.buildClient(), parserPool);

    }

    /**
     * Sets the OH entity ID
     * 
     * @param entityId
     */
    public void setEntityId(String entityId) {
        this.entityId= entityId;
    }

    /**
     * TODO EXPLAIN THE BLACK MAGIC!!!!!!
     * 
     * @param request
     * @param result
     * @return
     */
    protected Request mixObligationsInRequest(Request request, Result result) {
        ListIterator<Obligation> obligations_iterator= result.getObligations().listIterator();
        ListIterator<AttributeAssignment> aa_iterator= null;
        Set<Attribute> newEnvironmentAttributes= new HashSet<Attribute>();
        Obligation current_obligation= null;
        String trans_obligation_name= null;
        Request eesRequest= null;

        try {
            while (obligations_iterator.hasNext()) {
                Attribute transfer_obligations_attribute= new Attribute();
                current_obligation= obligations_iterator.next();
                aa_iterator= current_obligation.getAttributeAssignments().listIterator();

                transfer_obligations_attribute.setId(SPECIAL_EES_OBLIGATION_LIST_ATTR_ID);
                transfer_obligations_attribute.getValues().add(current_obligation.getId());
                transfer_obligations_attribute.setIssuer(getEntityId());
                newEnvironmentAttributes.add(transfer_obligations_attribute);

                while (aa_iterator.hasNext()) {
                    AttributeAssignment aa= aa_iterator.next();
                    Attribute a= new Attribute();

                    // TODO EXPLAIN THIS FORMAT !!!!

                    trans_obligation_name= URLEncoder.encode(current_obligation.getId(),
                                                             "UTF-8")
                            + "?attrId="
                            + URLEncoder.encode(aa.getAttributeId(), "UTF-8");
                    a.setId(trans_obligation_name);
                    a.setDataType(aa.getDataType());
                    a.setIssuer(getEntityId());
                    a.getValues().add(aa.getValue());
                    newEnvironmentAttributes.add(a);
                }
            }
            // copy the original request
            eesRequest= (Request) DeepCopy.copy(request);

            // add the new EES attribute in the environment
            Environment eesEnvironment= eesRequest.getEnvironment();
            eesEnvironment.getAttributes().addAll(newEnvironmentAttributes);

            // WARNING you are not really authorized to modify the request
            // request.setEnvironment(forwardedEnvironment);
        } catch (Exception e) {
            /* FIXME: who send an exception??? */
            log.error("EES ObligationHandler could not add in PDP obligations into EES request",
                      e);
        }

        if (log.isDebugEnabled()) {
            log.debug("Request after EES mix in of result\n{}",
                      eesRequest.toString());
        }

        return eesRequest;
    }

    /**
     * @return the OH entityID
     */
    public String getEntityId() {
        return entityId;
    }

    /**
     * @return the EES endpoint URL
     */
    public String getEESEndpoint() {
        return eesEndpoint;
    }

    /*
     * {@inheritDoc}
     * 
     * evaluates an incoming Request (from the PEP-client), which is merged with
     * a Result (from a PDP instance) by forwarding it to the EES and
     * interpreting its response
     */
    public boolean evaluateObligation(Request request, Result result)
            throws ObligationProcessingException {

        // only applied if there is a handled obligation in the result
        if (!resultContainsHandledObligationId(result)) {
            return false;
        }

        // mix the result in the request
        Request eesRequest= mixObligationsInRequest(request, result);

        // Send request with mixed in obligations to EES
        DecisionRequestContext messageContext= DecisionRequestContextHelper.buildMessageContext(getEntityId());
        sendRequestToEES(messageContext, eesRequest);

        // Receive list of remaining obligations
        Response eesResponse= extractResponseFromMessageContext(messageContext);
        Result eesResult= eesResponse.getResults().get(0);

        // Remove the handled obligations and replaced them with the ones
        // received from the EES server
        Iterator<Obligation> obligationItr= result.getObligations().iterator();
        List<Obligation> removedObligations= new ArrayList<Obligation>();
        while (obligationItr.hasNext()) {
            Obligation obligation= obligationItr.next();
            if (getObligationId().equals(obligation.getId())) {
                removedObligations.add(obligation);
            }
        }
        result.getObligations().removeAll(removedObligations);
        // and replace with the ones received by EES
        result.getObligations().addAll(eesResult.getObligations());

        return true;
    }

    /**
     * Returns <code>true</code> iff the result contains at least one handled
     * obligation
     * 
     * @param result
     *            the result
     * @return <code>true</code> iff the result contains at least one handled
     *         obligation
     */
    protected boolean resultContainsHandledObligationId(Result result) {
        for (Obligation obligation : result.getObligations()) {
            if (getObligationId().equals(obligation.getId())) {
                return true;
            }
        }
        log.debug("Result doesn't contain the handled obligationId: {}",
                  getObligationId());
        return false;
    }

    /**
     * TODO DOC extracts EES Response object from the message context inbound
     * message (SOAP response)
     * 
     * @param messageContext
     * @return
     * @throws EESObligationHandlerException
     */
    protected Response extractResponseFromMessageContext(
            DecisionRequestContext messageContext)
            throws EESObligationHandlerException {

        Envelope soapResponse= (Envelope) messageContext.getInboundMessage();
        if (soapResponse != null) {

            traceSOAPMessage(soapResponse, false);

            if (!soapResponse.getBody().getOrderedChildren().isEmpty()) {
                org.opensaml.saml2.core.Response samlResponse= (org.opensaml.saml2.core.Response) soapResponse.getBody().getOrderedChildren().get(0);
                if (!samlResponse.getAssertions().isEmpty()) {
                    Assertion samlAssertion= samlResponse.getAssertions().get(0);
                    List<Statement> xacmlDecisionStatements= samlAssertion.getStatements(XACMLAuthzDecisionStatementType.TYPE_NAME_XACML20);
                    if (!xacmlDecisionStatements.isEmpty()) {
                        XACMLAuthzDecisionStatementType xacmlDecisionStatement= (XACMLAuthzDecisionStatementType) xacmlDecisionStatements.get(0);
                        ResponseType xacmlResponse= xacmlDecisionStatement.getResponse();
                        if (xacmlResponse == null) {
                            String error= "No XACML Response found in Authorization Decision Statement";
                            log.error(error);
                            throw new EESObligationHandlerException(error);
                        }

                        RequestType xacmlRequest= xacmlDecisionStatement.getRequest();
                        /*
                        // not a problem if request is null
                        if (xacmlRequest == null) {
                            String error= "No effective XACML Request found in Authorization Decision Statement";
                            log.error(error);
                            throw new EESObligationHandlerException(error);
                        }
                        */
                        return XACMLConverter.responseFromXACML(xacmlResponse,
                                                                xacmlRequest);
                    }
                    else {
                        String error= "No XACML Authorization Decision Statement found in SAML Assertion";
                        log.error(error);
                        throw new EESObligationHandlerException(error);
                    }
                }
                else {
                    String error= "No SAML Assertion found in SAML Response";
                    log.error(error);
                    throw new EESObligationHandlerException(error);
                }
            }
            else {
                String error= "No SAML Reponse found in SOAP Body";
                log.error(error);
                throw new EESObligationHandlerException(error);
            }
        }
        else {
            String error= "No inbound SOAP Envelope found in message context";
            log.error(error);
            throw new EESObligationHandlerException(error);
        }
    }

    /**
     * Sends the "special" request to the EES server
     * 
     * @param messageContext
     *            the message context
     * @param request
     *            the request to send
     * @throws EESObligationHandlerException
     *             if an error occurs while sending the request
     */
    protected void sendRequestToEES(DecisionRequestContext messageContext,
            Request request) throws EESObligationHandlerException {

        RequestType xacmlRequest= XACMLConverter.requestToXACML(request);
        Envelope soapRequest= DecisionRequestContextHelper.buildSOAPMessage(getEntityId(),
                                                                            messageContext,
                                                                            xacmlRequest);
        traceSOAPMessage(soapRequest, true);

        try {
            log.debug("Send request {} to EES {}",
                      messageContext.getOutboundMessageId(),
                      getEESEndpoint());
            soapClient.send(getEESEndpoint(), messageContext);

        } catch (SOAPException e) {
            String error= "Error sending SOAP message to the EES "
                    + eesEndpoint + ": " + e.toString();
            log.error(error);
            throw new EESObligationHandlerException(error, e);
        } catch (SecurityException e) {
            // should never occur
            String error= "Error sending SOAP message to the EES "
                    + eesEndpoint + ": " + e.toString();
            log.error(error);
            throw new EESObligationHandlerException(error, e);
        }
    }

    /**
     * If log TRACE is enabled, then logs the SOAP message. The message will be
     * marshaled and pretty printed.
     * 
     * @param soapEnvelope
     *            The SOAP message envelope
     * @param isRequest
     *            true if it is a request, false if it is a response.
     */
    private void traceSOAPMessage(Envelope soapEnvelope, boolean isRequest) {
        if (log.isTraceEnabled()) {
            try {
                Element messageDom= Configuration.getMarshallerFactory().getMarshaller(soapEnvelope).marshall(soapEnvelope);
                if (isRequest) {
                    log.trace("Outgoing SOAP message\n{}",
                              XMLHelper.prettyPrintXML(messageDom));
                }
                else {
                    log.trace("Incoming SOAP message\n{}",
                              XMLHelper.prettyPrintXML(messageDom));
                }
            } catch (MarshallingException e) {
                log.error("Unable to marshall SOAP message: {}", e.getMessage());
            }
        }

    }

}
